/**
 * Module de validation des données
 */

const Validator = {
    /**
     * Valide une URL d'image
     * @param {string} url - URL à valider
     * @returns {boolean}
     */
    isValidImageUrl(url) {
        if (!url || typeof url !== 'string') {
            return false;
        }
        
        // Test regex pour URL valide
        const urlRegex = /^https?:\/\/.+\.(jpg|jpeg|png|gif|webp|bmp)(\?.*)?$/i;
        
        if (!urlRegex.test(url)) {
            console.warn('⚠️ URL invalide (format):', url);
            return false;
        }
        
        // Vérifier que l'URL est accessible (domaine valide)
        try {
            const urlObj = new URL(url);
            
            // Liste blanche des domaines autorisés (optionnel)
            const allowedDomains = [
                'img-conseil.duia.eu',
                'img-conseil.fr',
                'www.img-conseil.fr',
                // Ajouter vos domaines ici
            ];
            
            // Si liste blanche définie, vérifier
            if (allowedDomains.length > 0) {
                const isAllowed = allowedDomains.some(domain => 
                    urlObj.hostname === domain || urlObj.hostname.endsWith('.' + domain)
                );
                
                if (!isAllowed) {
                    console.warn('⚠️ Domaine non autorisé:', urlObj.hostname);
                    return false;
                }
            }
            
            return true;
        } catch (e) {
            console.warn('⚠️ URL malformée:', url);
            return false;
        }
    },
    
    /**
     * Valide la réponse du webhook de génération
     * @param {Object} response - Réponse du webhook
     * @returns {Object} { valid: boolean, error: string|null, data: Object|null }
     */
    validateGenerationResponse(response) {
        // Vérifier que la réponse existe
        if (!response) {
            return {
                valid: false,
                error: 'Aucune réponse du serveur',
                data: null
            };
        }
        
        // Vérifier le champ success
        if (response.success !== true) {
            return {
                valid: false,
                error: response.message || 'La génération a échoué',
                data: null
            };
        }
        
        // Vérifier que image_url existe
        if (!response.image_url) {
            return {
                valid: false,
                error: 'URL de l\'image manquante dans la réponse',
                data: null
            };
        }
        
        // Valider le format de l'URL
        if (!this.isValidImageUrl(response.image_url)) {
            return {
                valid: false,
                error: 'URL de l\'image invalide : ' + response.image_url,
                data: null
            };
        }
        
        // Tout est OK
        return {
            valid: true,
            error: null,
            data: {
                image_url: response.image_url,
                metadata: response.metadata || {}
            }
        };
    },
    
    /**
     * Teste si une URL d'image est accessible
     * @param {string} url - URL à tester
     * @returns {Promise<boolean>}
     */
    async testImageUrl(url) {
        return new Promise((resolve) => {
            const img = new Image();
            
            // Timeout de 10 secondes
            const timeout = setTimeout(() => {
                img.onload = null;
                img.onerror = null;
                console.warn('⚠️ Timeout lors du test de l\'image:', url);
                resolve(false);
            }, 10000);
            
            img.onload = () => {
                clearTimeout(timeout);
                console.log('✅ Image accessible:', url);
                resolve(true);
            };
            
            img.onerror = () => {
                clearTimeout(timeout);
                console.warn('❌ Image non accessible:', url);
                resolve(false);
            };
            
            img.src = url;
        });
    },
    
    /**
     * Valide un email
     * @param {string} email
     * @returns {boolean}
     */
    isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    },
    
    /**
     * Valide un numéro de téléphone français
     * @param {string} phone
     * @returns {boolean}
     */
    isValidPhone(phone) {
        const phoneRegex = /^(?:(?:\+|00)33|0)\s*[1-9](?:[\s.-]*\d{2}){4}$/;
        return phoneRegex.test(phone);
    },
    
    /**
     * Valide les données client pour la commande
     * @param {Object} customer
     * @returns {Object} { valid: boolean, errors: Array }
     */
    validateCustomerData(customer) {
        const errors = [];
        
        if (!customer.name || customer.name.trim().length < 2) {
            errors.push('Le nom est requis (minimum 2 caractères)');
        }
        
        if (!customer.prenom || customer.prenom.trim().length < 2) {
            errors.push('Le prénom est requis (minimum 2 caractères)');
        }
        
        if (!customer.email || !this.isValidEmail(customer.email)) {
            errors.push('L\'email est invalide');
        }
        
        if (!customer.phone || !this.isValidPhone(customer.phone)) {
            errors.push('Le numéro de téléphone est invalide');
        }
        
        if (!customer.address || customer.address.trim().length < 5) {
            errors.push('L\'adresse est requise (minimum 5 caractères)');
        }
        
        if (!customer.city || customer.city.trim().length < 2) {
            errors.push('La ville est requise');
        }
        
        if (!customer.zip || !/^\d{5}$/.test(customer.zip)) {
            errors.push('Le code postal doit contenir 5 chiffres');
        }
        
        return {
            valid: errors.length === 0,
            errors: errors
        };
    },
    
    /**
     * Nettoie et valide un prix
     * @param {*} price
     * @returns {number|null}
     */
    sanitizePrice(price) {
        const num = parseFloat(price);
        if (isNaN(num) || num < 0) {
            return null;
        }
        return Math.round(num * 100) / 100; // 2 décimales
    }
};
