/**
 * Gestionnaire d'état centralisé de l'application
 */

const StateManager = {
    // État initial
    state: {
        view: 'homepage',
        categories: [],
        allStyles: [],
        styles: [],
        selectedCategory: 'all',
        selectedStyle: null,
        uploadedFile: null,
        uploadedPreview: null,
        generatedImages: [],  // Tableau de versions { url, version }
        currentVersion: 0,     // Index de la version actuelle
        imageTitle: '',
        sceneDescription: '',
        backgroundType: 'ia',
        sessionId: '',
        generationCount: 0,
        format: 'A3',
        quantity: 1,
        deliveryMode: 'economique',
        customer: {
            name: '',
            prenom: '',
            email: '',
            phone: '',
            address: '',
            city: '',
            zip: ''
        },
        isSubmitting: false,
        cart: [],
        cartCount: 0,
        paymentId: null
    },
    
    // Listeners pour les changements d'état
    listeners: [],
    
    /**
     * Initialise l'état
     */
    init() {
        this.state.sessionId = this.generateSessionId();
        this.loadCartFromStorage();
    },
    
    /**
     * Génère un ID de session unique
     */
    generateSessionId() {
        const timestamp = Date.now().toString(36);
        const random = Math.random().toString(36).substring(2, 8);
        return `${timestamp}_${random}`;
    },
    
    /**
     * Met à jour l'état
     */
    setState(updates) {
        this.state = { ...this.state, ...updates };
        this.notifyListeners();
    },
    
    /**
     * Récupère l'état actuel
     */
    getState() {
        return this.state;
    },
    
    /**
     * Ajoute un listener
     */
    addListener(callback) {
        this.listeners.push(callback);
    },
    
    /**
     * Notifie les listeners
     */
    notifyListeners() {
        this.listeners.forEach(callback => callback(this.state));
    },
    
    /**
     * Charge le panier depuis le localStorage
     */
    loadCartFromStorage() {
        try {
            const savedCart = localStorage.getItem('cacartoon_cart');
            if (savedCart) {
                const cartData = JSON.parse(savedCart);
                this.state.cart = cartData.items || [];
                this.state.cartCount = this.state.cart.length;
                console.log('📂 Panier chargé:', this.state.cart.length, 'produits');
            } else {
                console.log('📂 Aucun panier sauvegardé');
            }
        } catch (e) {
            console.error('❌ Erreur chargement panier:', e);
            this.state.cart = [];
            this.state.cartCount = 0;
        }
    },
    
    /**
     * Sauvegarde le panier dans le localStorage
     */
    saveCartToStorage() {
        try {
            const cartData = {
                items: this.state.cart,
                count: this.state.cartCount,
                updated: new Date().toISOString()
            };
            localStorage.setItem('cacartoon_cart', JSON.stringify(cartData));
            console.log('💾 Panier sauvegardé:', cartData);
        } catch (e) {
            console.error('❌ Erreur sauvegarde panier:', e);
            UIManager.showToast('⚠️ Erreur de sauvegarde du panier');
        }
    },
    
    /**
     * Réinitialise l'état (pour retour à l'accueil)
     */
    resetToHome() {
        this.setState({
            view: 'homepage',
            selectedCategory: 'all',
            selectedStyle: null,
            uploadedFile: null,
            uploadedPreview: null,
            generatedImages: [],
            currentVersion: 0,
            imageTitle: '',
            sceneDescription: '',
            styles: this.state.allStyles,
            searchQuery: null,
            showingFavoris: false
        });
        document.body.classList.remove('order-page');
    }
};
