/**
 * Gestionnaire principal de rendu
 * Délègue le rendu aux modules spécifiques selon la vue
 */

const RenderManager = {
    lastView: null,
    
    // Vues qui ne doivent pas scroller lors d'un re-render
    noScrollOnRerender: ['result-generate', 'result', 'generating'],
    
    // Vues où le bandeau communication doit s'afficher
    homeViews: ['homepage', 'categories', 'styles'],
    
    /**
     * Fonction principale de rendu
     */
    render(options = {}) {
        const state = StateManager.getState();
        console.log('🎨 RENDER appelé - View:', state.view);
        
        const app = document.getElementById('app');
        if (!app) {
            console.error('❌ Element #app non trouvé');
            return;
        }
        
        // Gérer l'affichage du bandeau communication (uniquement sur la home)
        const commBanner = document.getElementById('commBanner');
        if (commBanner) {
            if (this.homeViews.includes(state.view)) {
                commBanner.style.display = 'block';
            } else {
                commBanner.style.display = 'none';
            }
        }
        
        // Gérer l'affichage des sections home-only (testimonials, newsletter, footer)
        const homeOnlySections = document.getElementById('homeOnlySections');
        if (homeOnlySections) {
            if (this.homeViews.includes(state.view)) {
                homeOnlySections.style.display = 'block';
            } else {
                homeOnlySections.style.display = 'none';
            }
        }
        
        // Scroll to top seulement si :
        // - La vue change ET
        // - On ne passe pas de 'generating' à 'result-generate' (même contexte)
        const shouldScroll = this.lastView !== state.view 
            && !(this.lastView === 'generating' && state.view === 'result-generate')
            && !options.noScroll;
            
        if (shouldScroll) {
            window.scrollTo({ top: 0, behavior: 'instant' });
        }
        
        this.lastView = state.view;
        
        // Déléguer le rendu selon la vue
        switch(state.view) {
            case 'homepage':
                ViewRenderers.renderHomepage(app, state);
                break;
            case 'categories':
                ViewRenderers.renderCategories(app, state);
                break;
            case 'styles':
                ViewRenderers.renderStyles(app, state);
                break;
            case 'upload':
                ViewRenderers.renderUpload(app, state);
                break;
            case 'result-generate':
                ViewRenderers.renderResultGenerate(app, state);
                break;
            case 'generating':
                ViewRenderers.renderGenerating(app, state);
                break;
            case 'result':
                ViewRenderers.renderResult(app, state);
                break;
            case 'format-selection':
                ViewRenderers.renderFormatSelection(app, state);
                break;
            case 'order':
                ViewRenderers.renderOrder(app, state);
                break;
            case 'cart':
                ViewRenderers.renderCart(app, state);
                break;
            case 'cart-added':
                ViewRenderers.renderCartAdded(app, state);
                break;
            case 'checkout':
                ViewRenderers.renderCheckout(app, state);
                break;
            case 'success':
                ViewRenderers.renderSuccess(app, state);
                break;
            default:
                console.error('❌ Vue inconnue:', state.view);
                this.renderError(app, 'Vue non trouvée');
        }
        
        // Réappliquer la protection des images après chaque rendu
        setTimeout(() => {
            if (typeof ImageProtection !== 'undefined') {
                ImageProtection.refresh();
            }
        }, 100);
    },
    
    /**
     * Affiche une erreur
     */
    renderError(app, message) {
        app.innerHTML = `
            <div class="error-page">
                <div class="error-icon">⚠️</div>
                <h1 class="error-title">Oups !</h1>
                <p class="error-message">${message}</p>
                <button class="btn btn-primary" onclick="NavigationManager.goHome()">
                    Retour à l'accueil
                </button>
            </div>
        `;
    }
};