// ===== MODALE DE PAIEMENT STRIPE =====

// Afficher la modale de paiement
function showPaymentModal(amount, onSuccess, onCancel) {
    const modalHTML = `
        <div id="paymentModal" class="payment-modal-overlay">
            <div class="payment-modal">
                <div class="payment-modal-header">
                    <h2>
                        <span>💳</span>
                        Paiement sécurisé
                    </h2>
                    <button class="payment-modal-close" onclick="closePaymentModal()">✕</button>
                </div>
                
                <div class="payment-modal-body">
                    <div class="payment-amount">
                        <div class="payment-amount-label">Montant à payer</div>
                        <div class="payment-amount-value">${Utils.formatPrice(amount)}</div>
                    </div>
                    
                    <form id="paymentForm" onsubmit="processPayment(event)">
                        <div class="payment-form-group">
                            <label>Numéro de carte *</label>
                            <div class="card-input-wrapper">
                                <input 
                                    type="text" 
                                    id="cardNumber" 
                                    class="payment-input" 
                                    placeholder="4242 4242 4242 4242"
                                    value="4242 4242 4242 4242"
                                    maxlength="19"
                                    required
                                    oninput="formatCardNumber(this)"
                                />
                                <span id="cardBrandIcon" class="card-brand-icon">💳</span>
                            </div>
                            <div id="cardNumberError" class="payment-error-message"></div>
                        </div>
                        
                        <div class="payment-form-row">
                            <div class="payment-form-group">
                                <label>Date d'expiration *</label>
                                <input 
                                    type="text" 
                                    id="cardExpiry" 
                                    class="payment-input" 
                                    placeholder="MM/AA"
                                    value="12/26"
                                    maxlength="5"
                                    required
                                    oninput="formatCardExpiry(this)"
                                />
                                <div id="cardExpiryError" class="payment-error-message"></div>
                            </div>
                            
                            <div class="payment-form-group">
                                <label>CVV *</label>
                                <input 
                                    type="text" 
                                    id="cardCVV" 
                                    class="payment-input" 
                                    placeholder="123"
                                    value="123"
                                    maxlength="4"
                                    required
                                    oninput="this.value = this.value.replace(/[^0-9]/g, '')"
                                />
                                <div id="cardCVVError" class="payment-error-message"></div>
                            </div>
                        </div>
                        
                        <div class="payment-form-group">
                            <label>Nom sur la carte *</label>
                            <input 
                                type="text" 
                                id="cardName" 
                                class="payment-input" 
                                placeholder="JEAN DUPONT"
                                value="TEST"
                                required
                                oninput="this.value = this.value.toUpperCase()"
                            />
                            <div id="cardNameError" class="payment-error-message"></div>
                        </div>
                        
                        <div class="payment-security-note">
                            🔒 Vos informations sont sécurisées et cryptées
                        </div>
                    </form>
                </div>
                
                <div class="payment-modal-footer">
                    <button type="button" class="payment-btn payment-btn-cancel" onclick="closePaymentModal()">
                        Annuler
                    </button>
                    <button type="submit" form="paymentForm" class="payment-btn payment-btn-submit" id="paymentSubmitBtn">
                        Payer ${Utils.formatPrice(amount)}
                    </button>
                </div>
                
                <div class="stripe-badge">
                    <span>Sécurisé par</span>
                    <svg class="stripe-logo" viewBox="0 0 60 25" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M59.3 10.3c0-5.8-2.9-10.3-8.4-10.3-5.5 0-8.9 4.5-8.9 10.2 0 6.7 3.8 10.2 9.2 10.2 2.7 0 4.7-.6 6.3-1.5v-4.7c-1.6.9-3.4 1.4-5.6 1.4-2.2 0-4.2-.8-4.5-3.6h11.8c0-.3.1-1.1.1-1.7zm-12-2.4c0-2.7 1.6-3.8 3.5-3.8 1.8 0 3.3 1.1 3.3 3.8h-6.8zM35.8 0c-2.2 0-3.6 1-4.4 1.7l-.3-1.4h-5.3v24.4l6-1.3V17c.8.6 2 1.4 4 1.4 4 0 7.7-3.2 7.7-10.3-.1-6.5-3.7-10.1-7.7-10.1zm-1.4 14c-1.3 0-2.1-.5-2.6-1.1V5.4c.6-.7 1.4-1.2 2.6-1.2 2 0 3.4 2.2 3.4 4.9 0 2.8-1.4 4.9-3.4 4.9zM19.5 1.6l6-1.3v-4.7l-6 1.3v4.7zM19.5.3h6v19.8h-6V.3zM10.9 2.6L10.6.3H5.3v19.8h6V7.8c1.4-1.8 3.8-1.5 4.5-1.2V.3c-.7-.3-3.4-.7-4.9 2.3z" fill="#6772E5"/>
                    </svg>
                </div>
            </div>
        </div>
    `;
    
    document.body.insertAdjacentHTML('beforeend', modalHTML);
    
    // Stocker les callbacks
    window.paymentModalSuccessCallback = onSuccess;
    window.paymentModalCancelCallback = onCancel;
    
    // Empêcher le scroll du body
    document.body.style.overflow = 'hidden';
    
    // Focus sur le premier champ
    setTimeout(() => {
        document.getElementById('cardNumber')?.focus();
    }, 100);
}

// Fermer la modale
function closePaymentModal(skipCancelCallback = false) {
    const modal = document.getElementById('paymentModal');
    if (modal) {
        modal.remove();
    }
    
    document.body.style.overflow = '';
    
    // N'appeler le callback d'annulation que si on n'a pas déjà eu un succès
    if (!skipCancelCallback && window.paymentModalCancelCallback) {
        window.paymentModalCancelCallback();
    }
    
    window.paymentModalSuccessCallback = null;
    window.paymentModalCancelCallback = null;
}

// Formater le numéro de carte
function formatCardNumber(input) {
    let value = input.value.replace(/\s/g, '').replace(/[^0-9]/g, '');
    let formatted = value.match(/.{1,4}/g)?.join(' ') || value;
    input.value = formatted;
    
    // Détecter la marque de carte
    const brandIcon = document.getElementById('cardBrandIcon');
    if (value.startsWith('4')) {
        brandIcon.textContent = '💳'; // Visa
    } else if (value.startsWith('5')) {
        brandIcon.textContent = '💳'; // Mastercard
    } else if (value.startsWith('3')) {
        brandIcon.textContent = '💳'; // Amex
    } else {
        brandIcon.textContent = '';
    }
}

// Formater la date d'expiration
function formatCardExpiry(input) {
    let value = input.value.replace(/\D/g, '');
    
    if (value.length >= 2) {
        value = value.slice(0, 2) + '/' + value.slice(2, 4);
    }
    
    input.value = value;
}

// Algorithme de Luhn pour valider les numéros de carte bancaire
function validateLuhn(cardNumber) {
    // Retirer tous les espaces et vérifier que ce sont bien des chiffres
    const digits = cardNumber.replace(/\s/g, '');
    if (!/^\d+$/.test(digits)) return false;
    
    let sum = 0;
    let isEven = false;
    
    // Parcourir de droite à gauche
    for (let i = digits.length - 1; i >= 0; i--) {
        let digit = parseInt(digits[i]);
        
        if (isEven) {
            digit *= 2;
            if (digit > 9) {
                digit -= 9;
            }
        }
        
        sum += digit;
        isEven = !isEven;
    }
    
    return (sum % 10) === 0;
}

// Déterminer le comportement selon la carte
function getCardTestBehavior(cardNumber) {
    const cleanNumber = cardNumber.replace(/\s/g, '');
    
    // Cartes de test avec comportements spécifiques
    const testCards = {
        // SUCCÈS
        '4242424242424242': { success: true, name: 'Visa Test - Succès' },
        '5555555555554444': { success: true, name: 'Mastercard Test - Succès' },
        '378282246310005': { success: true, name: 'Amex Test - Succès' },
        
        // REFUS - Fonds insuffisants
        '4000000000000002': { success: false, error: 'Fonds insuffisants' },
        
        // REFUS - Carte expirée
        '4000000000000069': { success: false, error: 'Carte expirée' },
        
        // REFUS - Code de sécurité incorrect
        '4000000000000127': { success: false, error: 'Code de sécurité incorrect' },
        
        // REFUS - Carte bloquée
        '4000000000000036': { success: false, error: 'Carte bloquée par votre banque' },
        
        // REFUS - Transaction refusée
        '4000000000000101': { success: false, error: 'Transaction refusée par votre banque' }
    };
    
    // Si c'est une carte de test connue, retourner son comportement
    if (testCards[cleanNumber]) {
        return testCards[cleanNumber];
    }
    
    // Sinon, simuler 95% de succès
    return Math.random() > 0.05 ? { success: true } : { success: false, error: 'Paiement refusé. Vérifiez vos informations.' };
}

// Valider la carte
function validateCard() {
    let isValid = true;
    
    // Numéro de carte
    const cardNumber = document.getElementById('cardNumber').value.replace(/\s/g, '');
    const cardNumberError = document.getElementById('cardNumberError');
    
    if (cardNumber.length < 13 || cardNumber.length > 19) {
        cardNumberError.textContent = 'Numéro de carte invalide';
        cardNumberError.classList.add('visible');
        document.getElementById('cardNumber').classList.add('error');
        isValid = false;
    } else if (!validateLuhn(cardNumber)) {
        cardNumberError.textContent = 'Numéro de carte invalide (vérification Luhn échouée)';
        cardNumberError.classList.add('visible');
        document.getElementById('cardNumber').classList.add('error');
        isValid = false;
    } else {
        cardNumberError.classList.remove('visible');
        document.getElementById('cardNumber').classList.remove('error');
    }
    
    // Date d'expiration
    const cardExpiry = document.getElementById('cardExpiry').value;
    const cardExpiryError = document.getElementById('cardExpiryError');
    const [month, year] = cardExpiry.split('/');
    
    if (!month || !year || parseInt(month) < 1 || parseInt(month) > 12) {
        cardExpiryError.textContent = 'Date invalide';
        cardExpiryError.classList.add('visible');
        document.getElementById('cardExpiry').classList.add('error');
        isValid = false;
    } else {
        // Vérifier que la carte n'est pas expirée
        // La carte est valide jusqu'à la fin du mois indiqué
        const now = new Date();
        const currentYear = now.getFullYear() % 100; // 2 derniers chiffres
        const currentMonth = now.getMonth() + 1; // Mois actuel (1-12)
        
        const cardYear = parseInt(year);
        const cardMonth = parseInt(month);
        
        // La carte est expirée si l'année est passée, ou si on est dans la même année mais le mois est passé
        const isExpired = (cardYear < currentYear) || (cardYear === currentYear && cardMonth < currentMonth);
        
        if (isExpired) {
            cardExpiryError.textContent = 'Carte expirée';
            cardExpiryError.classList.add('visible');
            document.getElementById('cardExpiry').classList.add('error');
            isValid = false;
        } else {
            cardExpiryError.classList.remove('visible');
            document.getElementById('cardExpiry').classList.remove('error');
        }
    }
    
    // CVV
    const cardCVV = document.getElementById('cardCVV').value;
    const cardCVVError = document.getElementById('cardCVVError');
    
    if (cardCVV.length < 3 || cardCVV.length > 4) {
        cardCVVError.textContent = 'CVV invalide';
        cardCVVError.classList.add('visible');
        document.getElementById('cardCVV').classList.add('error');
        isValid = false;
    } else {
        cardCVVError.classList.remove('visible');
        document.getElementById('cardCVV').classList.remove('error');
    }
    
    // Nom
    const cardName = document.getElementById('cardName').value.trim();
    const cardNameError = document.getElementById('cardNameError');
    
    if (cardName.length < 3) {
        cardNameError.textContent = 'Nom requis';
        cardNameError.classList.add('visible');
        document.getElementById('cardName').classList.add('error');
        isValid = false;
    } else {
        cardNameError.classList.remove('visible');
        document.getElementById('cardName').classList.remove('error');
    }
    
    return isValid;
}

// Traiter le paiement
async function processPayment(event) {
    event.preventDefault();
    
    console.log('🔍 DEBUG - processPayment appelé');
    
    const isValid = validateCard();
    console.log('Validation carte:', isValid);
    
    if (!isValid) {
        console.log('❌ Validation échouée');
        return;
    }
    
    console.log('✅ Validation OK - Début du traitement');
    
    // Récupérer le numéro de carte pour déterminer le comportement
    const cardNumber = document.getElementById('cardNumber').value;
    const cardBehavior = getCardTestBehavior(cardNumber);
    
    console.log('Carte utilisée:', cardBehavior.name || 'Carte inconnue');
    console.log('Comportement:', cardBehavior.success ? 'SUCCÈS' : 'ÉCHEC - ' + cardBehavior.error);
    
    const submitBtn = document.getElementById('paymentSubmitBtn');
    submitBtn.disabled = true;
    submitBtn.classList.add('processing');
    submitBtn.innerHTML = '<div class="payment-spinner"></div> Traitement en cours...';
    
    // Simuler le traitement du paiement (1-3 secondes)
    const delay = 1000 + Math.random() * 2000;
    
    setTimeout(() => {
        console.log('Résultat paiement:', cardBehavior.success ? 'SUCCÈS' : 'ÉCHEC');
        
        if (cardBehavior.success) {
            const paymentResult = {
                success: true,
                payment_id: 'pi_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9),
                amount: window.paymentAmount,
                currency: 'eur',
                status: 'succeeded',
                created: Date.now()
            };
            
            console.log('✅ Paiement réussi:', paymentResult);
            console.log('Callback existe:', !!window.paymentModalSuccessCallback);
            
            // IMPORTANT : Appeler le callback AVANT de fermer la modale
            if (window.paymentModalSuccessCallback) {
                window.paymentModalSuccessCallback(paymentResult);
            }
            
            // Fermer la modale APRÈS avoir appelé le callback
            closePaymentModal(true);
        } else {
            console.log('❌ Paiement refusé:', cardBehavior.error);
            submitBtn.disabled = false;
            submitBtn.classList.remove('processing');
            submitBtn.innerHTML = 'Réessayer';
            
            showToast('❌ ' + (cardBehavior.error || 'Paiement refusé. Vérifiez vos informations.'));
        }
    }, delay);
}
// Wrapper pour utilisation async/await
function openPaymentModal(amount) {
    return new Promise((resolve) => {
        showPaymentModal(
            amount,
            (paymentResult) => {
                // Succès
                resolve({ success: true, paymentResult });
            },
            () => {
                // Annulation
                resolve({ success: false, message: 'Paiement annulé' });
            }
        );
    });
}