/**
 * Protection des images contre la copie et le clic droit
 */

const ImageProtection = {
    /**
     * Initialise la protection globale des images
     */
    init() {
        console.log('🔒 Initialisation de la protection des images...');
        
        // Désactiver le clic droit sur toutes les images
        this.disableRightClickOnImages();
        
        // Désactiver le drag des images
        this.disableDragOnImages();
        
        // Désactiver certains raccourcis clavier
        this.disableKeyboardShortcuts();
        
        // Observer les nouvelles images ajoutées dynamiquement
        this.observeNewImages();
        
        console.log('✅ Protection des images activée');
    },
    
    /**
     * Désactive le clic droit sur toutes les images générées
     */
    disableRightClickOnImages() {
        // Méthode 1: Événement contextmenu sur le document
        document.addEventListener('contextmenu', (e) => {
            // Vérifier si l'élément cliqué est une image ou est dans un conteneur d'image
            const target = e.target;
            
            if (target.tagName === 'IMG' || 
                target.closest('.version-img') || 
                target.closest('.preview-image') ||
                target.closest('.watermarked-image') ||
                target.closest('.version-image-container')) {
                
                e.preventDefault();
                e.stopPropagation();
                
                // Afficher un message subtil
                Utils.showToast('🔒 Cette image est protégée', 1500);
                
                return false;
            }
        }, true);
        
        // Méthode 2: Appliquer directement aux images existantes
        this.applyProtectionToImages();
    },
    
    /**
     * Applique la protection aux images existantes
     */
    applyProtectionToImages() {
        const images = document.querySelectorAll('img.version-img, img.preview-image, #previewImage');
        
        images.forEach(img => {
            // Désactiver le clic droit
            img.addEventListener('contextmenu', (e) => {
                e.preventDefault();
                return false;
            });
            
            // Désactiver le drag
            img.addEventListener('dragstart', (e) => {
                e.preventDefault();
                return false;
            });
            
            // Rendre l'image non sélectionnable
            img.style.userSelect = 'none';
            img.style.webkitUserSelect = 'none';
            img.style.mozUserSelect = 'none';
            img.style.msUserSelect = 'none';
            
            // Désactiver le pointeur events pour éviter le drag
            img.setAttribute('draggable', 'false');
        });
    },
    
    /**
     * Désactive le drag and drop des images
     */
    disableDragOnImages() {
        document.addEventListener('dragstart', (e) => {
            if (e.target.tagName === 'IMG' && 
                (e.target.classList.contains('version-img') || 
                 e.target.classList.contains('preview-image') ||
                 e.target.id === 'previewImage')) {
                
                e.preventDefault();
                return false;
            }
        });
    },
    
    /**
     * Désactive certains raccourcis clavier pour éviter la copie
     */
    disableKeyboardShortcuts() {
        document.addEventListener('keydown', (e) => {
            // Ctrl+S (Sauvegarder)
            if ((e.ctrlKey || e.metaKey) && e.key === 's') {
                // Vérifier si on est sur une image
                const activeElement = document.activeElement;
                if (activeElement && activeElement.tagName === 'IMG') {
                    e.preventDefault();
                    Utils.showToast('🔒 Sauvegarde désactivée pour les images protégées', 1500);
                    return false;
                }
            }
            
            // Désactiver les outils de développeur sur les images (F12, Ctrl+Shift+I)
            // Note: Cela ne bloque pas complètement les dev tools mais limite l'accès facile
            if (e.key === 'F12' || 
                ((e.ctrlKey || e.metaKey) && e.shiftKey && (e.key === 'I' || e.key === 'J' || e.key === 'C'))) {
                
                const target = e.target;
                if (target.tagName === 'IMG' && 
                    (target.classList.contains('version-img') || 
                     target.classList.contains('preview-image'))) {
                    
                    e.preventDefault();
                    return false;
                }
            }
        });
    },
    
    /**
     * Observe l'ajout de nouvelles images dans le DOM pour leur appliquer la protection
     */
    observeNewImages() {
        const observer = new MutationObserver((mutations) => {
            mutations.forEach((mutation) => {
                mutation.addedNodes.forEach((node) => {
                    if (node.nodeType === 1) { // Element node
                        // Si c'est une image
                        if (node.tagName === 'IMG' && 
                            (node.classList.contains('version-img') || 
                             node.classList.contains('preview-image') ||
                             node.id === 'previewImage')) {
                            this.protectImage(node);
                        }
                        
                        // Si c'est un conteneur avec des images
                        const images = node.querySelectorAll('img.version-img, img.preview-image, #previewImage');
                        images.forEach(img => this.protectImage(img));
                    }
                });
            });
        });
        
        // Observer le conteneur principal de l'application
        const appContainer = document.getElementById('app');
        if (appContainer) {
            observer.observe(appContainer, {
                childList: true,
                subtree: true
            });
        }
        
        // Observer aussi la modale d'aperçu
        const previewModal = document.getElementById('imagePreviewModal');
        if (previewModal) {
            observer.observe(previewModal, {
                childList: true,
                subtree: true
            });
        }
    },
    
    /**
     * Protège une image spécifique
     */
    protectImage(img) {
        // Désactiver le clic droit
        img.addEventListener('contextmenu', (e) => {
            e.preventDefault();
            Utils.showToast('🔒 Cette image est protégée', 1500);
            return false;
        });
        
        // Désactiver le drag
        img.addEventListener('dragstart', (e) => {
            e.preventDefault();
            return false;
        });
        
        // CSS de protection
        img.style.userSelect = 'none';
        img.style.webkitUserSelect = 'none';
        img.style.mozUserSelect = 'none';
        img.style.msUserSelect = 'none';
        img.setAttribute('draggable', 'false');
        
        // Ajouter un attribut pour identifier les images protégées
        img.dataset.protected = 'true';
    },
    
    /**
     * Ajoute un filigrane renforcé sur une image
     */
    addWatermark(imageUrl, callback) {
        const canvas = document.createElement('canvas');
        const ctx = canvas.getContext('2d');
        const img = new Image();
        
        img.crossOrigin = 'Anonymous'; // Important pour éviter les erreurs CORS
        
        img.onload = () => {
            // Définir les dimensions du canvas
            canvas.width = img.width;
            canvas.height = img.height;
            
            // Dessiner l'image
            ctx.drawImage(img, 0, 0);
            
            // Configurer le filigrane
            ctx.font = 'bold 48px Arial';
            ctx.fillStyle = 'rgba(255, 255, 255, 0.4)';
            ctx.textAlign = 'center';
            ctx.textBaseline = 'middle';
            
            // Rotation
            ctx.save();
            ctx.translate(canvas.width / 2, canvas.height / 2);
            ctx.rotate(-45 * Math.PI / 180);
            
            // Dessiner le texte plusieurs fois pour couvrir l'image
            const text = 'Ça Cartoon - Ne pas copier';
            const spacing = 200;
            
            for (let y = -canvas.height; y < canvas.height * 2; y += spacing) {
                for (let x = -canvas.width; x < canvas.width * 2; x += spacing) {
                    ctx.fillText(text, x, y);
                }
            }
            
            ctx.restore();
            
            // Convertir en URL
            const watermarkedUrl = canvas.toDataURL('image/jpeg', 0.9);
            callback(watermarkedUrl);
        };
        
        img.src = imageUrl;
    },
    
    /**
     * Réinitialise la protection (utile après un changement de vue)
     */
    refresh() {
        this.applyProtectionToImages();
    }
};
