/**
 * Gestionnaire de la sélection des formats pour les images
 */

const FormatSelectionManager = {
    selectedImages: [],
    formatQuantities: {}, // {imageIndex: {format: quantity}}
    
    /**
     * Initialise le gestionnaire
     */
    init(images) {
        console.log('🎨 Initialisation FormatSelectionManager avec', images.length, 'images');
        this.selectedImages = images;
        this.formatQuantities = {};
        
        // Initialiser les quantités à 0
        images.forEach((_, index) => {
            this.formatQuantities[index] = {};
            Object.keys(CONFIG.FORMATS).forEach(format => {
                this.formatQuantities[index][format] = 0;
            });
        });
        
        // Mettre à jour l'affichage initial
        this.updateGlobalSummary();
    },
    
    /**
     * Mettre à jour le résumé global et le bouton
     */
    updateGlobalSummary() {
        let total = 0;
        let totalQty = 0;
        
        // Parcourir toutes les images et leurs quantités
        // La dégressivité s'applique par image + format
        Object.keys(this.formatQuantities).forEach(imageIndex => {
            const quantities = this.formatQuantities[imageIndex];
            Object.keys(quantities).forEach(format => {
                const qty = quantities[format];
                if (qty > 0) {
                    // Utiliser PricingManager.calculate pour appliquer la dégressivité
                    const pricing = PricingManager.calculate(qty, format);
                    total += pricing.total;
                    totalQty += qty;
                }
            });
        });
        
        // Mettre à jour l'affichage
        const summaryEl = document.getElementById('formatSummary');
        if (summaryEl) {
            summaryEl.querySelector('.summary-count').textContent = `${totalQty} article${totalQty > 1 ? 's' : ''}`;
            summaryEl.querySelector('.summary-total').textContent = Utils.formatPrice(total);
        }
        
        // Activer/désactiver le bouton global
        const btn = document.getElementById('btnAddAllToCart');
        if (btn) {
            btn.disabled = totalQty === 0;
            btn.innerHTML = totalQty > 0 
                ? `🛒 Ajouter ${totalQty} article${totalQty > 1 ? 's' : ''} au panier`
                : '🛒 Ajouter au panier';
        }
    },
    
    /**
     * Mettre à jour la quantité
     */
    updateQuantitySimple(imageIndex, format, value) {
        const qty = parseInt(value) || 0;
        const validQty = Math.max(0, Math.min(999, qty));
        
        this.formatQuantities[imageIndex][format] = validQty;
        
        // Highlight de la ligne
        const row = document.getElementById(`row-${imageIndex}-${format}`);
        if (row) {
            if (validQty > 0) {
                row.classList.add('selected');
            } else {
                row.classList.remove('selected');
            }
        }
        
        // Mettre à jour le résumé global
        this.updateGlobalSummary();
    },
    
    /**
     * Ajouter tous les formats de toutes les images au panier
     */
    addAllToCart() {
        const state = StateManager.getState();
        let totalAdded = 0;
        
        // Parcourir toutes les images
        Object.keys(this.formatQuantities).forEach(imageIndex => {
            const imageData = this.selectedImages[imageIndex];
            const quantities = this.formatQuantities[imageIndex] || {};
            
            Object.keys(quantities).forEach(format => {
                const qty = quantities[format];
                
                if (qty > 0) {
                    // Utiliser PricingManager.calculate pour appliquer la dégressivité
                    const pricing = PricingManager.calculate(qty, format);
                    
                    CartManager.addToCart({
                        generatedImage: imageData.image.url,
                        selectedStyle: state.selectedStyle,
                        format: format,
                        quantity: qty,
                        pricing: pricing,
                        version: imageData.image.version
                    }, false); // false = ne pas afficher de toast
                    
                    totalAdded += qty;
                }
            });
        });
        
        if (totalAdded > 0) {
            // Mettre à jour le compteur du panier
            CartManager.updateCartCounter();
            
            // Aller au panier
            StateManager.setState({ view: 'cart' });
            RenderManager.render();
        }
    },
    
    // Fonctions pour les boutons +/-
    incrementQuantity(imageIndex, format) {
        const input = document.getElementById(`qty-${imageIndex}-${format}`);
        if (!input) return;
        
        const currentValue = parseInt(input.value) || 0;
        const newValue = currentValue + 1;
        
        input.value = newValue;
        this.updateQuantitySimple(imageIndex, format, newValue);
    },
    
    decrementQuantity(imageIndex, format) {
        const input = document.getElementById(`qty-${imageIndex}-${format}`);
        if (!input) return;
        
        const currentValue = parseInt(input.value) || 0;
        const newValue = Math.max(0, currentValue - 1);
        
        input.value = newValue;
        this.updateQuantitySimple(imageIndex, format, newValue);
    }
};
