/**
 * Gestionnaire des favoris
 */

const FavorisManager = {
    favoris: [], // Liste des style_id favoris
    
    /**
     * Récupère l'utilisateur connecté
     */
    getUser() {
        if (typeof authManager !== 'undefined' && authManager.isAuthenticated && authManager.user) {
            return authManager.user;
        }
        return null;
    },
    
    /**
     * Récupère l'ID du client connecté
     */
    getClientId() {
        const user = this.getUser();
        if (user) {
            // Essayer différentes propriétés possibles
            return user.id || user.client_id || null;
        }
        return null;
    },
    
    /**
     * Vide les favoris (utilisé lors de la déconnexion)
     */
    clear() {
        this.favoris = [];
        localStorage.removeItem('cacartoon_favoris');
        this.updateHeaderBadge();
        console.log('🗑️ Favoris vidés');
    },
    
    /**
     * Initialise les favoris (charger depuis l'API si connecté)
     */
    async init() {
        const clientId = this.getClientId();
        if (clientId) {
            await this.loadFavoris(clientId);
        } else {
            // Charger depuis localStorage si non connecté
            this.loadFromLocalStorage();
        }
        this.updateHeaderBadge();
    },
    
    /**
     * Charge les favoris depuis l'API
     */
    async loadFavoris(clientId) {
        try {
            console.log('📥 Chargement favoris pour client_id:', clientId);
            const response = await fetch(`${CONFIG.API_URL}?action=favoris&client_id=${clientId}`);
            const data = await response.json();
            
            console.log('📥 Réponse API favoris:', data);
            
            if (data.success) {
                // Récupérer les style_id depuis les données
                this.favoris = data.data.map(f => {
                    // Le style_id peut être un nombre ou une chaîne
                    const id = f.style_id;
                    return typeof id === 'string' ? parseInt(id, 10) : id;
                });
                console.log('❤️ Favoris chargés:', this.favoris.length, this.favoris);
                this.updateHeaderBadge();
            } else {
                console.error('❌ Erreur API favoris:', data.error);
            }
        } catch (error) {
            console.error('❌ Erreur chargement favoris:', error);
            this.loadFromLocalStorage();
        }
    },
    
    /**
     * Charge depuis localStorage (pour utilisateurs non connectés)
     */
    loadFromLocalStorage() {
        try {
            const saved = localStorage.getItem('cacartoon_favoris');
            if (saved) {
                this.favoris = JSON.parse(saved);
            }
        } catch (e) {
            this.favoris = [];
        }
    },
    
    /**
     * Sauvegarde dans localStorage
     */
    saveToLocalStorage() {
        try {
            localStorage.setItem('cacartoon_favoris', JSON.stringify(this.favoris));
        } catch (e) {
            console.error('Erreur sauvegarde favoris:', e);
        }
    },
    
    /**
     * Vérifie si un style est en favori
     */
    isFavori(styleId) {
        return this.favoris.includes(styleId);
    },
    
    /**
     * Toggle favori (ajoute ou supprime)
     */
    async toggleFavori(styleId, event) {
        if (event) {
            event.stopPropagation();
            event.preventDefault();
        }
        
        const clientId = this.getClientId();
        
        if (clientId) {
            // Utilisateur connecté - sync avec API
            try {
                console.log('📤 Toggle favori - client_id:', clientId, 'style_id:', styleId);
                
                const response = await fetch(`${CONFIG.API_URL}?action=toggle_favori`, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        client_id: clientId,
                        style_id: styleId
                    })
                });
                
                const data = await response.json();
                console.log('📥 Réponse toggle favori:', data);
                
                if (data.success) {
                    if (data.is_favori) {
                        this.favoris.push(styleId);
                        Utils.showToast('❤️ Ajouté aux favoris');
                    } else {
                        this.favoris = this.favoris.filter(id => id !== styleId);
                        Utils.showToast('💔 Retiré des favoris');
                    }
                } else {
                    console.error('❌ Erreur API:', data.error);
                    Utils.showToast('❌ Erreur: ' + (data.error || 'Erreur inconnue'));
                }
            } catch (error) {
                console.error('Erreur toggle favori:', error);
                Utils.showToast('❌ Erreur de connexion');
            }
        } else {
            // Non connecté - localStorage uniquement
            if (this.isFavori(styleId)) {
                this.favoris = this.favoris.filter(id => id !== styleId);
                Utils.showToast('💔 Retiré des favoris');
            } else {
                this.favoris.push(styleId);
                Utils.showToast('❤️ Ajouté aux favoris');
            }
            this.saveToLocalStorage();
        }
        
        // Mettre à jour l'UI
        this.updateHeartIcon(styleId);
        this.updateHeaderBadge();
    },
    
    /**
     * Met à jour l'icône cœur d'un style
     */
    updateHeartIcon(styleId) {
        const heart = document.querySelector(`[data-favori-style="${styleId}"]`);
        if (heart) {
            if (this.isFavori(styleId)) {
                heart.classList.add('active');
                heart.innerHTML = '❤️';
                heart.title = 'Retirer des favoris';
            } else {
                heart.classList.remove('active');
                heart.innerHTML = '🤍';
                heart.title = 'Ajouter aux favoris';
            }
        }
    },
    
    /**
     * Met à jour le badge du header
     */
    updateHeaderBadge() {
        const badge = document.getElementById('favoris-count');
        if (badge) {
            const count = this.favoris.length;
            badge.textContent = count;
            badge.style.display = count > 0 ? 'flex' : 'none';
        }
    },
    
    /**
     * Affiche les favoris (appelé depuis le header)
     */
    async showFavoris() {
        const user = this.getUser();
        
        // Recharger les favoris si connecté
        if (user && user.client_id) {
            await this.loadFavoris(user.client_id);
        }
        
        if (this.favoris.length === 0) {
            // Pas de favoris - afficher un message
            this.showEmptyFavoris();
            return;
        }
        
        // Charger les styles favoris
        try {
            const state = StateManager.getState();
            let favorisStyles = [];
            
            if (user && user.client_id) {
                // Depuis l'API
                const response = await fetch(`${CONFIG.API_URL}?action=favoris&client_id=${user.client_id}`);
                const data = await response.json();
                if (data.success) {
                    favorisStyles = data.data;
                }
            } else {
                // Depuis allStyles
                favorisStyles = state.allStyles.filter(s => this.favoris.includes(s.style_id));
            }
            
            // Afficher les favoris
            StateManager.setState({
                view: 'homepage',
                styles: favorisStyles,
                selectedCategory: null,
                searchQuery: null,
                showingFavoris: true
            });
            
            RenderManager.render();
            
            // Afficher le titre
            setTimeout(() => {
                const title = document.querySelector('.section-title');
                if (title) {
                    title.innerHTML = `❤️ Mes favoris <span style="font-size: 0.6em; color: var(--text-muted);">(${favorisStyles.length})</span>`;
                }
            }, 50);
            
        } catch (error) {
            console.error('Erreur affichage favoris:', error);
            Utils.showToast('❌ Erreur lors du chargement des favoris');
        }
    },
    
    /**
     * Affiche un message si pas de favoris
     */
    showEmptyFavoris() {
        const app = document.getElementById('app');
        app.innerHTML = `
            <div class="empty-favoris-container fade-in">
                <div class="empty-favoris-content">
                    <div class="empty-favoris-icon">💔</div>
                    <h2>Pas encore de favoris</h2>
                    <p>Cliquez sur le cœur 🤍 d'un style pour l'ajouter à vos favoris</p>
                    <button class="btn btn-primary" onclick="NavigationManager.goHome()">
                        Découvrir les styles
                    </button>
                </div>
            </div>
        `;
    },
    
    /**
     * Génère le HTML du cœur pour un style
     */
    getHeartHTML(styleId) {
        const isFav = this.isFavori(styleId);
        return `
            <button class="favori-heart ${isFav ? 'active' : ''}" 
                    data-favori-style="${styleId}"
                    onclick="FavorisManager.toggleFavori(${styleId}, event)"
                    title="${isFav ? 'Retirer des favoris' : 'Ajouter aux favoris'}">
                ${isFav ? '❤️' : '🤍'}
            </button>
        `;
    }
};

// Exposer globalement
window.FavorisManager = FavorisManager;
