/**
 * Gestionnaire du panier d'achat
 */

const CartManager = {
    /**
     * Ajoute un article au panier (fusionne si même image + format)
     */
    addToCart(product, showToastMsg = false) {
        const state = StateManager.getState();
        
        // Chercher si un article identique existe déjà (même image + même format)
        const existingItem = state.cart.find(item => 
            item.image === product.generatedImage && 
            item.format === product.format
        );
        
        if (existingItem) {
            // Fusionner : ajouter la quantité
            const newQuantity = existingItem.quantity + product.quantity;
            this.updateQuantity(existingItem.id, newQuantity);
            
            console.log('🔄 Article fusionné, nouvelle quantité:', newQuantity);
            
            if (showToastMsg) {
                Utils.showToast(`✅ Quantité mise à jour (${newQuantity} ${product.format})`);
            }
        } else {
            // Nouvel article
            const cartItem = {
                id: Utils.generateId(),
                image: product.generatedImage,
                styleName: product.selectedStyle?.titre || product.selectedStyle?.name || 'Style personnalisé',
                format: product.format,
                formatLabel: CONFIG.FORMATS[product.format].label,
                formatDim: CONFIG.FORMATS[product.format].dim,
                quantity: product.quantity,
                unitPrice: product.pricing.unitaire,
                totalPrice: product.pricing.total,
                addedAt: new Date().toISOString()
            };
            
            state.cart.push(cartItem);
            state.cartCount = state.cart.length;
            
            StateManager.setState({ 
                cart: state.cart, 
                cartCount: state.cartCount 
            });
            StateManager.saveCartToStorage();
            
            console.log('✅ Article ajouté:', cartItem);
            
            if (showToastMsg) {
                Utils.showToast('✅ Produit ajouté au panier');
            }
        }
        
        console.log('📊 Panier total:', state.cart.length, 'lignes');
        this.updateCartCounter();
    },
    
    /**
     * Retire un article du panier
     */
    removeFromCart(itemId) {
        const state = StateManager.getState();
        const itemIndex = state.cart.findIndex(item => item.id === itemId);
        
        if (itemIndex > -1) {
            state.cart.splice(itemIndex, 1);
            state.cartCount = state.cart.length;
            
            StateManager.setState({ 
                cart: state.cart, 
                cartCount: state.cartCount 
            });
            StateManager.saveCartToStorage();
            
            Utils.showToast('🗑️ Article retiré du panier');
            this.updateCartCounter();
        }
    },
    
    /**
     * Incrémente la quantité d'un article
     */
    incrementQuantity(itemId) {
        const state = StateManager.getState();
        const item = state.cart.find(item => item.id === itemId);
        
        if (item && item.quantity < 1000) {
            this.updateQuantity(itemId, item.quantity + 1);
        }
    },
    
    /**
     * Décrémente la quantité d'un article
     */
    decrementQuantity(itemId) {
        const state = StateManager.getState();
        const item = state.cart.find(item => item.id === itemId);
        
        if (item && item.quantity > 1) {
            this.updateQuantity(itemId, item.quantity - 1);
        }
    },
    
    /**
     * Met à jour la quantité d'un article
     */
    updateQuantity(itemId, newQuantity) {
        const state = StateManager.getState();
        const item = state.cart.find(item => item.id === itemId);
        
        if (item) {
            item.quantity = Math.max(1, Math.min(1000, parseInt(newQuantity)));
            
            const pricing = PricingManager.calculate(
                item.quantity, 
                item.format
            );
            
            item.unitPrice = pricing.unitaire;
            item.totalPrice = pricing.total;
            
            StateManager.setState({ cart: state.cart });
            StateManager.saveCartToStorage();
        }
    },
    
    /**
     * Vide le panier
     */
    clearCart() {
        if (confirm('Voulez-vous vraiment vider le panier ?')) {
            StateManager.setState({
                cart: [],
                cartCount: 0
            });
            StateManager.saveCartToStorage();
            Utils.showToast('🗑️ Panier vidé');
            this.updateCartCounter();
            NavigationManager.goHome();
        }
    },
    
    /**
     * Calcule le total du panier
     */
    getTotal() {
        const state = StateManager.getState();
        return state.cart.reduce((sum, item) => sum + item.totalPrice, 0);
    },
    
    /**
     * Calcule le nombre total d'unités
     */
    getTotalUnits() {
        const state = StateManager.getState();
        return state.cart.reduce((sum, item) => sum + item.quantity, 0);
    },
    
    /**
     * Retourne le nombre d'articles dans le panier
     */
    getCartCount() {
        const state = StateManager.getState();
        return state.cart ? state.cart.length : 0;
    },
    
    /**
     * Met à jour le compteur visuel du panier
     */
    updateCartCounter() {
        const state = StateManager.getState();
        const cartCountEl = document.getElementById('cartCount');
        
        if (cartCountEl) {
            cartCountEl.textContent = state.cartCount;
            
            if (state.cartCount === 0) {
                cartCountEl.classList.add('empty');
            } else {
                cartCountEl.classList.remove('empty');
            }
            
            // Animation
            cartCountEl.style.transform = 'scale(1.3)';
            setTimeout(() => {
                cartCountEl.style.transform = 'scale(1)';
            }, 200);
        }
    },
    
    /**
     * Affiche la vue panier
     */
    viewCart() {
        StateManager.setState({ view: 'cart' });
    },
    
    // =====================================================
    // GESTION DES CODES PROMO
    // =====================================================
    
    promoCode: null,       // Code promo actif
    promoReduction: 0,     // Montant de la réduction
    promoData: null,       // Données du code promo
    
    /**
     * Applique un code promo
     */
    async applyPromoCode(code) {
        // Masquer le message d'erreur précédent
        this.hidePromoError();
        
        if (!code || code.trim() === '') {
            this.showPromoError('Veuillez entrer un code promo');
            return false;
        }
        
        const total = this.getTotal();
        
        try {
            const response = await fetch(`${CONFIG.API_URL}?action=verify_promo`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    code: code.trim(),
                    montant: total
                })
            });
            
            const data = await response.json();
            
            if (data.success) {
                this.promoCode = data.data.code;
                this.promoReduction = data.data.reduction;
                this.promoData = data.data;
                
                // Masquer l'erreur et mettre à jour l'affichage
                this.hidePromoError();
                this.updatePromoDisplay();
                
                return true;
            } else {
                // Message d'erreur personnalisé
                let errorMsg = 'Code erroné ou mal saisi';
                if (data.error) {
                    if (data.error.includes('invalide')) {
                        errorMsg = 'Code erroné ou mal saisi';
                    } else if (data.error.includes('expiré')) {
                        errorMsg = 'Ce code promo a expiré';
                    } else if (data.error.includes('actif')) {
                        errorMsg = 'Ce code promo n\'est plus valide';
                    } else if (data.error.includes('limite')) {
                        errorMsg = 'Ce code promo n\'est plus disponible';
                    } else if (data.error.includes('minimum')) {
                        errorMsg = data.error;
                    } else {
                        errorMsg = data.error;
                    }
                }
                this.showPromoError(errorMsg);
                return false;
            }
        } catch (error) {
            this.showPromoError('Code erroné ou mal saisi');
            return false;
        }
    },
    
    /**
     * Affiche un message d'erreur sous le champ code promo
     */
    showPromoError(message) {
        const errorDiv = document.getElementById('promoErrorMsg');
        if (errorDiv) {
            errorDiv.textContent = '❌ ' + message;
            errorDiv.style.display = 'block';
        }
        // Aussi en console
        console.log('❌', message);
    },
    
    /**
     * Masque le message d'erreur
     */
    hidePromoError() {
        const errorDiv = document.getElementById('promoErrorMsg');
        if (errorDiv) {
            errorDiv.style.display = 'none';
            errorDiv.textContent = '';
        }
    },
    
    /**
     * Supprime le code promo
     */
    removePromoCode() {
        this.promoCode = null;
        this.promoReduction = 0;
        this.promoData = null;
        this.updatePromoDisplay();
    },
    
    /**
     * Retourne le total avec réduction
     */
    getTotalWithPromo() {
        const total = this.getTotal();
        return Math.max(0, total - this.promoReduction);
    },
    
    /**
     * Met à jour l'affichage du code promo dans le panier
     */
    updatePromoDisplay() {
        const promoContainer = document.getElementById('promoCodeDisplay');
        const totalElement = document.getElementById('cartTotalAmount');
        const totalFinalElement = document.getElementById('cartTotalFinal');
        
        if (promoContainer) {
            if (this.promoCode) {
                promoContainer.innerHTML = `
                    <div class="promo-applied">
                        <span class="promo-tag">🎉 ${this.promoCode}</span>
                        <span class="promo-discount">-${this.promoReduction.toFixed(2)} €</span>
                        <button class="promo-remove" onclick="CartManager.removePromoCode()">✕</button>
                    </div>
                `;
                promoContainer.style.display = 'block';
            } else {
                promoContainer.innerHTML = '';
                promoContainer.style.display = 'none';
            }
        }
        
        // Mettre à jour les totaux
        if (totalElement) {
            const total = this.getTotal();
            if (this.promoCode) {
                totalElement.innerHTML = `<s style="color:#999">${total.toFixed(2)} €</s>`;
            } else {
                totalElement.textContent = `${total.toFixed(2)} €`;
            }
        }
        
        if (totalFinalElement) {
            totalFinalElement.textContent = `${this.getTotalWithPromo().toFixed(2)} €`;
            if (this.promoCode) {
                totalFinalElement.classList.add('with-promo');
            } else {
                totalFinalElement.classList.remove('with-promo');
            }
        }
    },
    
    /**
     * Recalcule la promo si le panier change
     */
    async recalculatePromo() {
        if (this.promoCode) {
            await this.applyPromoCode(this.promoCode);
        }
    }
};
