/**
 * Gestionnaire des appels API
 */

const ApiManager = {
    /**
     * Charge toutes les catégories
     */
    async loadCategories() {
        try {
            const response = await fetch(`${CONFIG.API_URL}?action=categories`);
            const data = await response.json();
            if (data.success) {
                return data.data;
            }
            throw new Error('Erreur lors du chargement des catégories');
        } catch (error) {
            console.error('❌ Erreur loadCategories:', error);
            throw error;
        }
    },
    
    /**
     * Charge tous les styles
     */
    async loadAllStyles() {
        try {
            const response = await fetch(`${CONFIG.API_URL}?action=styles`);
            const data = await response.json();
            if (data.success) {
                return data.data;
            }
            throw new Error('Erreur lors du chargement des styles');
        } catch (error) {
            console.error('❌ Erreur loadAllStyles:', error);
            throw error;
        }
    },
    
    /**
     * Charge les styles d'une catégorie
     */
    async loadStylesByCategory(categorie) {
        try {
            const response = await fetch(`${CONFIG.API_URL}?action=styles&categorie=${categorie}`);
            const data = await response.json();
            if (data.success) {
                return data.data;
            }
            throw new Error('Erreur lors du chargement des styles');
        } catch (error) {
            console.error('❌ Erreur loadStylesByCategory:', error);
            throw error;
        }
    },
    
    /**
     * Recherche de styles par mot-clé
     */
    async searchStyles(query) {
        try {
            const response = await fetch(`${CONFIG.API_URL}?action=search_styles&q=${encodeURIComponent(query)}`);
            const data = await response.json();
            if (data.success) {
                return data.data;
            }
            throw new Error('Erreur lors de la recherche');
        } catch (error) {
            console.error('❌ Erreur searchStyles:', error);
            throw error;
        }
    },
    
    /**
     * Charge un style spécifique
     */
    async loadStyle(styleId) {
        try {
            const response = await fetch(`${CONFIG.API_URL}?action=style&id=${styleId}`);
            const data = await response.json();
            if (data.success) {
                return data.data;
            }
            throw new Error('Style non trouvé');
        } catch (error) {
            console.error('❌ Erreur loadStyle:', error);
            throw error;
        }
    },
    
    /**
     * Génère une image avec l'IA
     */
    async generateImage(formData) {
        try {
            console.log('📤 Envoi au webhook:', CONFIG.WEBHOOK_GENERATE);
            
            const response = await fetch(CONFIG.WEBHOOK_GENERATE, {
                method: 'POST',
                body: formData
            });
            
            console.log('📥 Réponse HTTP:', response.status, response.statusText);
            
            if (!response.ok) {
                const errorText = await response.text();
                console.error('❌ Réponse erreur:', errorText);
                throw new Error('Erreur serveur lors de la génération');
            }
            
            const data = await response.json();
            console.log('📦 Données reçues:', data);
            
            if (!data.success) {
                console.error('❌ Success = false, message:', data.message);
                throw new Error(data.message || 'Erreur lors de la génération');
            }
            
            console.log('✅ Image générée:', data.image_url);
            return data;
        } catch (error) {
            console.error('❌ Erreur generateImage:', error);
            throw error;
        }
    },
    
    /**
     * Envoie une commande (webhook + BDD locale)
     */
    async submitOrder(formData) {
        try {
            // 1. Sauvegarder en BDD locale via l'API
            const apiResponse = await fetch(`${CONFIG.API_URL}?action=submit_order`, {
                method: 'POST',
                body: formData
            });
            
            const apiResult = await apiResponse.json();
            
            if (!apiResult.success) {
                console.warn('⚠️ Erreur sauvegarde BDD:', apiResult.error);
            } else {
                console.log('✅ Commande sauvegardée en BDD:', apiResult.data);
            }
            
            // 2. Envoyer au webhook (notifications, etc.)
            try {
                const webhookResponse = await fetch(CONFIG.WEBHOOK_ORDER, {
                    method: 'POST',
                    body: formData
                });
                
                if (!webhookResponse.ok) {
                    console.warn('⚠️ Webhook non disponible');
                }
            } catch (webhookError) {
                console.warn('⚠️ Webhook error:', webhookError);
                // On ne bloque pas si le webhook échoue
            }
            
            return apiResult;
        } catch (error) {
            console.error('❌ Erreur submitOrder:', error);
            throw error;
        }
    },
    
    /**
     * Met à jour les statistiques client
     */
    async updateClientStats(montant) {
        try {
            if (!authManager || !authManager.isAuthenticated) {
                return false;
            }
            
            const response = await fetch('update_client_stats.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${authManager.token}`
                },
                body: JSON.stringify({ montant })
            });
            
            if (response.ok) {
                const data = await response.json();
                console.log('✅ Stats mises à jour:', data);
                return true;
            }
            
            return false;
        } catch (error) {
            console.error('❌ Erreur updateClientStats:', error);
            return false;
        }
    }
};
