<?php
/**
 * ADMINISTRATION POSTERS PERSONNALISÉS
 * Version PostgreSQL - AVEC UPLOAD IMAGE + AUTO-INCREMENT
 * Style graphique sombre moderne
 */

// Configuration des erreurs
error_reporting(E_ALL);
ini_set('display_errors', '0');
ini_set('log_errors', '1');
ini_set('error_log', __DIR__ . '/manage_style_errors.log');

// ============================================
// CONFIGURATION BASE DE DONNÉES PostgreSQL
// ============================================
define('DB_HOST', 'oz9643-001.eu.clouddb.ovh.net');
define('DB_PORT', '35297');
define('DB_USER', 'user_recreatoon');
define('DB_PASS', 'MathelodIMG59');
define('DB_NAME', 'recreatoon_DDB');

// ============================================
// AUTHENTIFICATION
// ============================================
session_start();

function checkAuth() {
    try {
        $dsn = sprintf("pgsql:host=%s;port=%s;dbname=%s", DB_HOST, DB_PORT, DB_NAME);
        $pdo = new PDO($dsn, DB_USER, DB_PASS, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
        ]);
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login']) && isset($_POST['password'])) {
            $login = $_POST['login'];
            $password = $_POST['password'];
            
            $stmt = $pdo->prepare('SELECT * FROM auth_access WHERE login = ? AND mdp = ?');
            $stmt->execute([$login, $password]);
            $user = $stmt->fetch();
            
            if ($user) {
                $_SESSION['auth'] = true;
                $_SESSION['login'] = $login;
                return true;
            } else {
                return false;
            }
        }
        
        return isset($_SESSION['auth']) && $_SESSION['auth'] === true;
    } catch (PDOException $e) {
        error_log("Auth error: " . $e->getMessage());
        return false;
    }
}

// Déconnexion
if (isset($_GET['logout'])) {
    session_destroy();
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit;
}

// Vérifier l'authentification (sauf pour les actions API qui ont leur propre sécurité)
$action = $_GET['action'] ?? '';
$isApiCall = !empty($action);

if (!$isApiCall && !checkAuth()) {
    $error = '';
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login'])) {
        $error = 'Login ou mot de passe incorrect';
    }
    ?>
    <!DOCTYPE html>
    <html lang="fr">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Connexion - Administration</title>
        <style>
            * { box-sizing: border-box; margin: 0; padding: 0; }
            body {
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
                min-height: 100vh;
                display: flex;
                align-items: center;
                justify-content: center;
            }
            .login-container {
                background: rgba(255,255,255,0.05);
                border: 1px solid rgba(255,255,255,0.1);
                border-radius: 16px;
                padding: 40px;
                width: 100%;
                max-width: 400px;
                backdrop-filter: blur(10px);
            }
            h1 {
                color: #fff;
                text-align: center;
                margin-bottom: 30px;
                font-size: 1.8rem;
            }
            .form-group {
                margin-bottom: 20px;
            }
            label {
                display: block;
                color: #a0a0a0;
                margin-bottom: 8px;
                font-size: 0.9rem;
            }
            input {
                width: 100%;
                padding: 12px 16px;
                border: 1px solid rgba(255,255,255,0.2);
                border-radius: 8px;
                background: rgba(255,255,255,0.05);
                color: #fff;
                font-size: 1rem;
            }
            input:focus {
                outline: none;
                border-color: #6366f1;
            }
            button {
                width: 100%;
                padding: 14px;
                background: linear-gradient(135deg, #6366f1, #8b5cf6);
                border: none;
                border-radius: 8px;
                color: #fff;
                font-size: 1rem;
                font-weight: 600;
                cursor: pointer;
                transition: transform 0.2s, box-shadow 0.2s;
            }
            button:hover {
                transform: translateY(-2px);
                box-shadow: 0 10px 20px rgba(99, 102, 241, 0.3);
            }
            .error {
                background: rgba(239, 68, 68, 0.2);
                border: 1px solid rgba(239, 68, 68, 0.5);
                color: #fca5a5;
                padding: 12px;
                border-radius: 8px;
                margin-bottom: 20px;
                text-align: center;
            }
        </style>
    </head>
    <body>
        <div class="login-container">
            <h1>🔐 Administration</h1>
            <?php if ($error): ?>
                <div class="error"><?= htmlspecialchars($error) ?></div>
            <?php endif; ?>
            <form method="POST">
                <div class="form-group">
                    <label>Login</label>
                    <input type="text" name="login" required autofocus>
                </div>
                <div class="form-group">
                    <label>Mot de passe</label>
                    <input type="password" name="password" required>
                </div>
                <button type="submit">Se connecter</button>
            </form>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// Configuration upload d'images - CHEMIN RELATIF
define('UPLOAD_DIR', __DIR__ . '/../styles/');
define('UPLOAD_URL', '/styles/');
define('MAX_FILE_SIZE', 5 * 1024 * 1024); // 5 MB
define('ALLOWED_TYPES', ['image/jpeg', 'image/png', 'image/webp', 'image/gif']);

// Créer le dossier d'upload s'il n'existe pas (sans warning)
if (!is_dir(UPLOAD_DIR)) {
    @mkdir(UPLOAD_DIR, 0755, true);
}

// Connexion PDO PostgreSQL
function getDB() {
    static $pdo = null;
    if ($pdo === null) {
        try {
            $dsn = sprintf(
                "pgsql:host=%s;port=%s;dbname=%s",
                DB_HOST,
                DB_PORT,
                DB_NAME
            );
            $pdo = new PDO($dsn, DB_USER, DB_PASS, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false
            ]);
        } catch(PDOException $e) {
            error_log("Erreur DB: " . $e->getMessage());
            http_response_code(500);
            header('Content-Type: application/json');
            die(json_encode([
                'success' => false, 
                'error' => 'Connexion base de données échouée: ' . $e->getMessage()
            ]));
        }
    }
    return $pdo;
}

// Initialisation de la table si nécessaire
function initTable() {
    $pdo = getDB();
    try {
        // Table categories
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS categories (
                id SERIAL PRIMARY KEY,
                slug VARCHAR(100) UNIQUE NOT NULL,
                nom VARCHAR(255) NOT NULL,
                description TEXT,
                url_img_categorie VARCHAR(500) DEFAULT '',
                ordre INTEGER DEFAULT 0,
                actif BOOLEAN DEFAULT TRUE,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ");
        
        // Table styles
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS styles (
                id SERIAL PRIMARY KEY,
                style_id INTEGER NOT NULL,
                titre VARCHAR(255) NOT NULL,
                theme VARCHAR(100) DEFAULT 'Personnage',
                categorie_slug VARCHAR(100) NOT NULL,
                categorie_nom VARCHAR(255) NOT NULL,
                descriptif TEXT NOT NULL,
                prompt TEXT NOT NULL,
                background VARCHAR(50) DEFAULT 'auto',
                url VARCHAR(500) DEFAULT '',
                ordre INTEGER DEFAULT 0,
                actif BOOLEAN DEFAULT TRUE,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ");
        
        $pdo->exec("CREATE INDEX IF NOT EXISTS idx_categorie ON styles(categorie_slug)");
        $pdo->exec("CREATE INDEX IF NOT EXISTS idx_actif ON styles(actif)");
        $pdo->exec("CREATE INDEX IF NOT EXISTS idx_style_id ON styles(style_id)");
        
    } catch(PDOException $e) {
        error_log("Erreur init table: " . $e->getMessage());
    }
}

// Fonction pour obtenir le prochain style_id
function getNextStyleId() {
    $pdo = getDB();
    try {
        $stmt = $pdo->query('SELECT COALESCE(MAX(style_id), 0) + 1 as next_id FROM styles');
        $result = $stmt->fetch();
        return (int)$result['next_id'];
    } catch(PDOException $e) {
        error_log("Erreur getNextStyleId: " . $e->getMessage());
        return 1;
    }
}

// Fonction pour uploader une image
function uploadImage($file, $titre = '') {
    // Vérifier les erreurs
    if ($file['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('Erreur lors du téléchargement du fichier');
    }
    
    // Vérifier la taille
    if ($file['size'] > MAX_FILE_SIZE) {
        throw new Exception('Le fichier est trop volumineux (max 5 MB)');
    }
    
    // Vérifier le type MIME
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mimeType = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);
    
    if (!in_array($mimeType, ALLOWED_TYPES)) {
        throw new Exception('Type de fichier non autorisé. Utilisez JPG, PNG, WebP ou GIF');
    }
    
    // Générer un nom de fichier basé sur le titre
    $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    
    // Si un titre est fourni, l'utiliser, sinon générer un nom unique
    if (!empty($titre)) {
        // Nettoyer le titre : remplacer espaces par underscores, garder alphanumériques
        $cleanTitle = preg_replace('/[^a-zA-Z0-9_-]/', '_', $titre);
        $cleanTitle = preg_replace('/_+/', '_', $cleanTitle); // Éviter les underscores multiples
        $cleanTitle = trim($cleanTitle, '_');
        $filename = $cleanTitle . '.' . $extension;
    } else {
        // Nom unique si pas de titre
        $filename = 'style_' . uniqid() . '.' . $extension;
    }
    
    $filepath = UPLOAD_DIR . $filename;
    
    // Si le fichier existe déjà, le supprimer
    if (file_exists($filepath)) {
        unlink($filepath);
    }
    
    // Déplacer le fichier
    if (!move_uploaded_file($file['tmp_name'], $filepath)) {
        throw new Exception('Impossible de sauvegarder le fichier');
    }
    
    // Retourner l'URL relative
    return UPLOAD_URL . $filename;
}

// ============================================
// API REST - TRAITEMENT AVANT LE HTML
// ============================================
$action = $_GET['action'] ?? '';

if ($action) {
    // Initialiser la table AVANT toute action API
    initTable();
    
    header('Content-Type: application/json; charset=utf-8');
    $pdo = getDB();
    $method = $_SERVER['REQUEST_METHOD'];
    
    try {
        switch($action) {
            
            case 'test':
                echo json_encode([
                    'success' => true, 
                    'message' => 'Connexion PostgreSQL OK',
                    'version' => $pdo->query('SELECT version()')->fetchColumn()
                ]);
                break;
            
            case 'get_next_id':
                echo json_encode([
                    'success' => true,
                    'next_id' => getNextStyleId()
                ]);
                break;
            
            case 'upload_image':
                if ($method !== 'POST') {
                    throw new Exception('Méthode non autorisée');
                }
                
                if (!isset($_FILES['image'])) {
                    throw new Exception('Aucune image fournie');
                }
                
                $titre = $_POST['titre'] ?? '';
                $url = uploadImage($_FILES['image'], $titre);
                
                echo json_encode([
                    'success' => true,
                    'url' => $url
                ]);
                break;
            
            case 'upload_category_image':
                if ($method !== 'POST') {
                    throw new Exception('Méthode non autorisée');
                }
                
                if (!isset($_FILES['image'])) {
                    throw new Exception('Aucune image fournie');
                }
                
                $categorySlug = $_POST['category_slug'] ?? '';
                $url = uploadImage($_FILES['image'], 'category_' . $categorySlug);
                
                echo json_encode([
                    'success' => true,
                    'url' => $url
                ]);
                break;
            
            case 'list':
                $stmt = $pdo->query('
                    SELECT * FROM styles 
                    ORDER BY categorie_nom, ordre, style_id
                ');
                echo json_encode([
                    'success' => true, 
                    'data' => $stmt->fetchAll()
                ]);
                break;
            
            case 'list_categories':
                $stmt = $pdo->query('
                    SELECT 
                        c.slug,
                        c.nom,
                        c.description,
                        c.url_img_categorie,
                        c.ordre,
                        c.actif,
                        COUNT(s.id) as nb_styles,
                        COUNT(CASE WHEN s.actif = TRUE THEN 1 END) as nb_styles_actifs
                    FROM categories c
                    LEFT JOIN styles s ON c.slug = s.categorie_slug
                    GROUP BY c.slug, c.nom, c.description, c.url_img_categorie, c.ordre, c.actif
                    ORDER BY c.ordre, c.nom
                ');
                echo json_encode([
                    'success' => true, 
                    'data' => $stmt->fetchAll()
                ]);
                break;
            
            case 'get':
                $id = intval($_GET['id'] ?? 0);
                if ($id <= 0) {
                    throw new Exception('ID invalide');
                }
                $stmt = $pdo->prepare('SELECT * FROM styles WHERE id = ?');
                $stmt->execute([$id]);
                $result = $stmt->fetch();
                echo json_encode([
                    'success' => true, 
                    'data' => $result ?: null
                ]);
                break;
            
            case 'create':
                if ($method !== 'POST') {
                    throw new Exception('Méthode non autorisée');
                }
                
                $data = json_decode(file_get_contents('php://input'), true);
                
                if (!$data) {
                    throw new Exception('Données JSON invalides');
                }
                
                // Debug: logger les données reçues
                error_log('DEBUG CREATE - Data reçue: ' . json_encode($data));
                error_log('DEBUG CREATE - flag_champ_scene: ' . ($data['flag_champ_scene'] ?? 'NOT SET'));
                
                if (!isset($data['style_id']) || $data['style_id'] <= 0) {
                    $data['style_id'] = getNextStyleId();
                }
                
                $stmt = $pdo->prepare('
                    INSERT INTO styles (
                        style_id, titre, theme, categorie_slug, categorie_nom, 
                        descriptif, prompt, background, url, url2, ordre, actif, flag_champ_scene
                    ) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                    RETURNING *
                ');
                
                $stmt->execute([
                    $data['style_id'],
                    $data['titre'] ?? '',
                    $data['theme'] ?? 'Personnage',
                    $data['categorie_slug'] ?? '',
                    $data['categorie_nom'] ?? '',
                    $data['descriptif'] ?? '',
                    $data['prompt'] ?? '',
                    $data['background'] ?? 'auto',
                    $data['url'] ?? '',
                    $data['url2'] ?? '',
                    $data['ordre'] ?? 0,
                    isset($data['actif']) ? ($data['actif'] ? 't' : 'f') : 't',
                    isset($data['flag_champ_scene']) ? ($data['flag_champ_scene'] ? 't' : 'f') : 'f'
                ]);
                
                $result = $stmt->fetch();
                echo json_encode([
                    'success' => true, 
                    'id' => $result['id'],
                    'style_id' => $result['style_id'],
                    'data' => $result
                ]);
                break;
            
            case 'update':
                if ($method !== 'POST') {
                    throw new Exception('Méthode non autorisée');
                }
                
                $data = json_decode(file_get_contents('php://input'), true);
                
                if (!$data || !isset($data['id'])) {
                    throw new Exception('Données invalides');
                }
                
                $stmt = $pdo->prepare('
                    UPDATE styles SET 
                        style_id = ?, 
                        titre = ?, 
                        theme = ?, 
                        categorie_slug = ?, 
                        categorie_nom = ?, 
                        descriptif = ?, 
                        prompt = ?,
                        background = ?, 
                        url = ?,
                        url2 = ?, 
                        ordre = ?, 
                        actif = ?,
                        flag_champ_scene = ?,
                        updated_at = CURRENT_TIMESTAMP
                    WHERE id = ?
                ');
                
                $stmt->execute([
                    $data['style_id'],
                    $data['titre'],
                    $data['theme'] ?? 'Personnage',
                    $data['categorie_slug'],
                    $data['categorie_nom'],
                    $data['descriptif'],
                    $data['prompt'],
                    $data['background'] ?? 'auto',
                    $data['url'] ?? '',
                    $data['url2'] ?? '',
                    $data['ordre'] ?? 0,
                    $data['actif'] ? 't' : 'f',
                    isset($data['flag_champ_scene']) ? ($data['flag_champ_scene'] ? 't' : 'f') : 'f',
                    $data['id']
                ]);
                
                echo json_encode([
                    'success' => true,
                    'affected' => $stmt->rowCount()
                ]);
                break;
            
            case 'delete':
                $id = intval($_GET['id'] ?? 0);
                if ($id <= 0) {
                    throw new Exception('ID invalide');
                }
                
                $stmt = $pdo->prepare('DELETE FROM styles WHERE id = ?');
                $stmt->execute([$id]);
                
                echo json_encode([
                    'success' => true,
                    'affected' => $stmt->rowCount()
                ]);
                break;
            
            case 'toggle':
                if ($method !== 'POST') {
                    throw new Exception('Méthode non autorisée');
                }
                
                $id = intval($_GET['id'] ?? 0);
                if ($id <= 0) {
                    throw new Exception('ID invalide');
                }
                
                $stmt = $pdo->prepare('
                    UPDATE styles 
                    SET actif = NOT actif, updated_at = CURRENT_TIMESTAMP 
                    WHERE id = ? 
                    RETURNING actif
                ');
                $stmt->execute([$id]);
                $result = $stmt->fetch();
                
                echo json_encode([
                    'success' => true,
                    'actif' => $result['actif']
                ]);
                break;
            
            case 'toggle_category':
                if ($method !== 'POST') {
                    throw new Exception('Méthode non autorisée');
                }
                
                $slug = $_GET['slug'] ?? '';
                if (empty($slug)) {
                    throw new Exception('Slug invalide');
                }
                
                $stmt = $pdo->prepare('
                    UPDATE categories 
                    SET actif = NOT actif, updated_at = CURRENT_TIMESTAMP 
                    WHERE slug = ? 
                    RETURNING actif
                ');
                $stmt->execute([$slug]);
                $result = $stmt->fetch();
                
                echo json_encode([
                    'success' => true,
                    'actif' => $result['actif']
                ]);
                break;
            
            case 'save_order':
                if ($method !== 'POST') {
                    throw new Exception('Méthode non autorisée');
                }
                
                $data = json_decode(file_get_contents('php://input'), true);
                
                if (!isset($data['orders']) || !is_array($data['orders'])) {
                    throw new Exception('Données invalides');
                }
                
                $pdo->beginTransaction();
                
                try {
                    $stmt = $pdo->prepare('UPDATE styles SET ordre = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?');
                    
                    foreach ($data['orders'] as $item) {
                        $stmt->execute([$item['ordre'], $item['id']]);
                    }
                    
                    $pdo->commit();
                    
                    echo json_encode([
                        'success' => true,
                        'message' => 'Ordre sauvegardé',
                        'count' => count($data['orders'])
                    ]);
                } catch (Exception $e) {
                    $pdo->rollBack();
                    throw $e;
                }
                break;
            
            case 'stats':
                $total = $pdo->query('SELECT COUNT(*) FROM styles')->fetchColumn();
                $actifs = $pdo->query('SELECT COUNT(*) FROM styles WHERE actif = TRUE')->fetchColumn();
                $categories = $pdo->query('SELECT COUNT(DISTINCT categorie_slug) FROM styles')->fetchColumn();
                
                echo json_encode([
                    'success' => true,
                    'stats' => [
                        'total' => (int)$total,
                        'actifs' => (int)$actifs,
                        'categories' => (int)$categories
                    ]
                ]);
                break;
            
            case 'import':
                if ($method !== 'POST') {
                    throw new Exception('Méthode non autorisée');
                }
                
                $json = json_decode(file_get_contents('php://input'), true);
                
                if (!isset($json['categories'])) {
                    throw new Exception('Format JSON invalide');
                }
                
                $pdo->beginTransaction();
                
                try {
                    $cat_stmt = $pdo->prepare('
                        INSERT INTO categories (slug, nom, description, ordre, actif)
                        VALUES (?, ?, ?, ?, TRUE)
                        ON CONFLICT (slug) 
                        DO UPDATE SET 
                            nom = EXCLUDED.nom,
                            description = EXCLUDED.description,
                            ordre = EXCLUDED.ordre,
                            actif = TRUE
                    ');
                    
                    $cat_count = 0;
                    $cat_ordre = 1;
                    
                    foreach ($json['categories'] as $category) {
                        $description = $category['description'] ?? 
                                      "Collection de styles pour " . strtolower($category['nom']);
                        
                        $cat_stmt->execute([
                            $category['id'],
                            $category['nom'],
                            $description,
                            $cat_ordre++
                        ]);
                        $cat_count++;
                    }
                    
                    $pdo->exec('TRUNCATE TABLE styles RESTART IDENTITY');
                    
                    $style_count = 0;
                    $style_ordre = 0;
                    
                    $style_stmt = $pdo->prepare('
                        INSERT INTO styles (
                            style_id, titre, theme, categorie_slug, categorie_nom, 
                            descriptif, prompt, background, url, ordre, actif
                        ) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, TRUE)
                    ');
                    
                    foreach ($json['categories'] as $category) {
                        foreach ($category['styles'] as $style) {
                            $style_stmt->execute([
                                $style['id'] ?? 1,
                                $style['titre'] ?? '',
                                $json['theme'] ?? 'Personnage',
                                $category['id'] ?? '',
                                $category['nom'] ?? '',
                                $style['descriptif'] ?? '',
                                $style['prompt'] ?? '',
                                $style['background'] ?? 'auto',
                                $style['url'] ?? '',
                                $style_ordre++
                            ]);
                            $style_count++;
                        }
                    }
                    
                    $pdo->commit();
                    echo json_encode([
                        'success' => true, 
                        'imported_categories' => $cat_count,
                        'imported_styles' => $style_count,
                        'message' => "✅ {$cat_count} catégories et {$style_count} styles importés avec succès"
                    ]);
                    
                } catch(Exception $e) {
                    $pdo->rollBack();
                    throw $e;
                }
                break;
            
            case 'export':
                $stmt = $pdo->query('
                    SELECT DISTINCT categorie_slug, categorie_nom 
                    FROM styles 
                    ORDER BY categorie_nom
                ');
                
                $categories = [];
                
                foreach ($stmt->fetchAll() as $cat) {
                    $styles_stmt = $pdo->prepare('
                        SELECT 
                            style_id as id, 
                            titre, 
                            descriptif, 
                            prompt, 
                            url 
                        FROM styles 
                        WHERE categorie_slug = ? AND actif = TRUE 
                        ORDER BY ordre
                    ');
                    $styles_stmt->execute([$cat['categorie_slug']]);
                    
                    $categories[] = [
                        'id' => $cat['categorie_slug'],
                        'nom' => $cat['categorie_nom'],
                        'styles' => $styles_stmt->fetchAll()
                    ];
                }
                
                $export = [
                    'theme' => 'Personnage',
                    'categories' => $categories,
                    'metadata' => [
                        'version' => '2.0',
                        'exported_at' => date('Y-m-d H:i:s')
                    ]
                ];
                
                echo json_encode($export, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
                break;
            
            case 'update_category':
                if ($method !== 'POST') {
                    throw new Exception('Méthode non autorisée');
                }
                
                $data = json_decode(file_get_contents('php://input'), true);
                
                if (!$data || !isset($data['old_slug']) || !isset($data['new_nom'])) {
                    throw new Exception('Données invalides');
                }
                
                $stmt = $pdo->prepare('
                    UPDATE categories SET 
                        nom = ?,
                        url_img_categorie = ?,
                        updated_at = CURRENT_TIMESTAMP
                    WHERE slug = ?
                ');
                $stmt->execute([
                    $data['new_nom'],
                    $data['url_img_categorie'] ?? '',
                    $data['old_slug']
                ]);
                
                $stmt = $pdo->prepare('
                    UPDATE styles SET 
                        categorie_nom = ?,
                        updated_at = CURRENT_TIMESTAMP
                    WHERE categorie_slug = ?
                ');
                
                $stmt->execute([
                    $data['new_nom'],
                    $data['old_slug']
                ]);
                
                echo json_encode([
                    'success' => true,
                    'affected' => $stmt->rowCount()
                ]);
                break;
            
            case 'create_category':
                if ($method !== 'POST') {
                    throw new Exception('Méthode non autorisée');
                }
                
                $data = json_decode(file_get_contents('php://input'), true);
                
                if (!$data || !isset($data['slug']) || !isset($data['nom'])) {
                    throw new Exception('Données invalides');
                }
                
                $stmt = $pdo->prepare('
                    INSERT INTO categories (slug, nom, description, url_img_categorie, ordre, actif)
                    VALUES (?, ?, ?, ?, ?, TRUE)
                    RETURNING id
                ');
                
                $stmt->execute([
                    $data['slug'],
                    $data['nom'],
                    $data['description'] ?? '',
                    $data['url_img_categorie'] ?? '',
                    $data['ordre'] ?? 0
                ]);
                
                $result = $stmt->fetch();
                echo json_encode([
                    'success' => true,
                    'id' => $result['id']
                ]);
                break;
            
            case 'delete_category':
                $slug = $_GET['slug'] ?? '';
                if (empty($slug)) {
                    throw new Exception('Slug invalide');
                }
                
                $stmt = $pdo->prepare('DELETE FROM categories WHERE slug = ?');
                $stmt->execute([$slug]);
                
                echo json_encode([
                    'success' => true,
                    'affected' => $stmt->rowCount()
                ]);
                break;
            
            default:
                http_response_code(404);
                echo json_encode([
                    'success' => false, 
                    'error' => 'Action non trouvée'
                ]);
        }
        
    } catch(PDOException $e) {
        http_response_code(500);
        error_log("Erreur SQL: " . $e->getMessage());
        echo json_encode([
            'success' => false, 
            'error' => 'Erreur de base de données: ' . $e->getMessage()
        ]);
    } catch(Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false, 
            'error' => $e->getMessage()
        ]);
    }
    exit;
}

// Initialiser la table au premier chargement de l'interface
initTable();
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Posters - PostgreSQL</title>
    <link href="https://fonts.googleapis.com/css2?family=Bebas+Neue&family=Work+Sans:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="manage_style.css">
</head>
<body>
    <div class="container">
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1rem;">
            <h1>🎨 ADMINISTRATION POSTERS <span class="pg-badge">PostgreSQL</span></h1>
            <a href="?logout=1" style="background: rgba(239,68,68,0.2); border: 1px solid rgba(239,68,68,0.5); color: #fca5a5; padding: 8px 16px; border-radius: 8px; text-decoration: none; font-size: 0.9rem;">🚪 Déconnexion (<?= htmlspecialchars($_SESSION['login'] ?? '') ?>)</a>
        </div>
        
        <div id="connectionStatus" class="connection-status">⏳ Vérification connexion...</div>
        
        <div id="alertSuccess" class="alert alert-success"></div>
        <div id="alertError" class="alert alert-error"></div>
        
        <div class="stats">
            <div class="stat-card">
                <div class="stat-value" id="statTotal">-</div>
                <div class="stat-label">Total Styles</div>
            </div>
            <div class="stat-card">
                <div class="stat-value" id="statActifs">-</div>
                <div class="stat-label">Styles Actifs</div>
            </div>
            <div class="stat-card">
                <div class="stat-value" id="statCategories">-</div>
                <div class="stat-label">Catégories</div>
            </div>
        </div>
        
        <div class="tabs">
            <button class="tab active" onclick="showTab('styles')">🎨 Styles</button>
            <button class="tab" onclick="showTab('categories')">📁 Catégories</button>
            <button class="tab" onclick="showTab('ordre')">↕️ Gérer l'ordre</button>
            <button class="tab" onclick="showTab('import')">📥 Import/Export</button>
        </div>
        
        <div id="tab-styles" class="tab-content active">
            <div class="card">
                <div class="card-header">
                    <h2>📋 Liste des Styles</h2>
                    <div style="display: flex; gap: 12px;">
                        <button class="btn btn-secondary btn-sm" onclick="clearFilters()">🔄 Réinitialiser</button>
                        <button class="btn btn-primary" onclick="openModal()">➕ Nouveau Style</button>
                    </div>
                </div>
                <div id="stylesTable">Chargement...</div>
            </div>
        </div>
        
        <div id="tab-categories" class="tab-content">
            <div class="card">
                <div class="card-header">
                    <h2>📁 Gestion des Catégories</h2>
                    <button class="btn btn-primary" onclick="openCategoryModal()">➕ Nouvelle Catégorie</button>
                </div>
                <div id="categoriesTable">Chargement...</div>
            </div>
        </div>
        
        <div id="tab-ordre" class="tab-content">
            <div class="card">
                <div class="card-header">
                    <h2>↕️ Gérer l'ordre des styles</h2>
                    <div style="display: flex; gap: 12px; align-items: center;">
                        <select id="ordreFilterCategory" onchange="loadOrdreList()" style="padding: 8px 12px; border-radius: 8px; background: var(--bg-dark); border: 1px solid var(--border); color: var(--text-light);">
                            <option value="all">Toutes les catégories</option>
                        </select>
                        <label style="display: flex; align-items: center; gap: 8px; color: var(--text-muted); font-size: 0.9rem;">
                            <input type="checkbox" id="ordreShowActiveOnly" checked onchange="loadOrdreList()" style="width: auto;">
                            Actifs uniquement
                        </label>
                        <button class="btn btn-success" onclick="saveAllOrders()">💾 Sauvegarder l'ordre</button>
                    </div>
                </div>
                <p style="color: var(--text-muted); margin: 1rem 0; font-size: 0.9rem;">
                    ⬆️⬇️ Utilisez les boutons pour déplacer les styles. L'ordre le plus petit apparaît en premier.
                </p>
                <div id="ordreList" style="display: flex; flex-direction: column; gap: 4px;">Chargement...</div>
            </div>
        </div>
        
        <div id="tab-import" class="tab-content">
            <div class="card">
                <h3 style="margin-bottom: 1.5rem;">📥 Import JSON</h3>
                <div class="file-upload" onclick="document.getElementById('jsonFile').click()">
                    <input type="file" id="jsonFile" accept=".json" style="display:none" onchange="importJSON(this)">
                    <p style="font-size: 4rem; margin-bottom: 1rem;">📂</p>
                    <p style="font-size: 1.1rem; font-weight: 600; color: var(--text-light); margin-bottom: 0.5rem;">Cliquez pour sélectionner un fichier JSON</p>
                    <p>Remplace toutes les données existantes</p>
                </div>
            </div>
            <div class="card">
                <h3 style="margin-bottom: 1.5rem;">💾 Export JSON</h3>
                <p style="color: var(--text-muted); margin-bottom: 1.5rem;">Télécharger tous les styles actifs au format JSON</p>
                <button class="btn btn-success" onclick="exportJSON()">💾 Télécharger JSON</button>
            </div>
        </div>
    </div>
    
    <!-- MODAL STYLE -->
    <div id="modal" class="modal">
        <div class="modal-content">
            <h3 id="modalTitle">✨ Nouveau Style</h3>
            <form id="styleForm" onsubmit="saveStyle(event)">
                <input type="hidden" id="styleDbId">
                
                <div class="form-group">
                    <label>ID Style <span class="required">*</span></label>
                    <input type="number" id="styleId" min="1" readonly style="background: var(--bg-dark); opacity: 0.7;">
                    <small>Auto-incrémenté automatiquement</small>
                </div>
                
                <div class="form-group">
                    <label>Titre <span class="required">*</span></label>
                    <input type="text" id="styleTitre" required placeholder="Ex: Affiche Vintage">
                </div>
                
                <div class="form-group">
                    <label>Catégorie <span class="required">*</span></label>
                    <select id="styleCategorie" required>
                        <option value="">-- Sélectionner une catégorie --</option>
                    </select>
                    <input type="hidden" id="categorieSlug">
                    <input type="hidden" id="categorieNom">
                </div>
                
                <div class="form-group">
                    <label>Image du style</label>
                    <div class="image-upload-zone" id="uploadZone" onclick="document.getElementById('imageFile').click()">
                        <input type="file" id="imageFile" accept="image/*" style="display:none" onchange="handleImageSelect(this)">
                        <p style="font-size: 2.5rem; margin-bottom: 0.5rem;">📷</p>
                        <p style="font-weight: 600; color: var(--text-light);">Cliquez ou glissez une image</p>
                        <p style="font-size: 0.85rem;">JPG, PNG, WebP ou GIF (max 5 MB)</p>
                    </div>
                    <div class="upload-progress" id="uploadProgress">
                        <div class="progress-bar">
                            <div class="progress-fill" id="progressFill"></div>
                        </div>
                    </div>
                    <img id="imagePreview" class="image-preview" style="display:none;">
                </div>
                
                <div class="form-group">
                    <label>URL Image</label>
                    <input type="text" id="styleUrl" placeholder="Rempli automatiquement après upload" readonly style="opacity: 0.7;">
                    <small>Ce champ se remplit automatiquement après l'upload</small>
                </div>
                
                <div class="form-group">
                    <label>Image 2 du style (optionnel)</label>
                    <div class="image-upload-zone" id="uploadZone2" onclick="document.getElementById('imageFile2').click()" style="border-color: var(--accent);">
                        <input type="file" id="imageFile2" accept="image/*" style="display:none" onchange="handleImageSelect2(this)">
                        <p style="font-size: 2.5rem; margin-bottom: 0.5rem;">📷</p>
                        <p style="font-weight: 600; color: var(--text-light);">Cliquez ou glissez une 2ème image</p>
                        <p style="font-size: 0.85rem;">JPG, PNG, WebP ou GIF (max 5 MB)</p>
                    </div>
                    <div class="upload-progress" id="uploadProgress2">
                        <div class="progress-bar">
                            <div class="progress-fill" id="progressFill2"></div>
                        </div>
                    </div>
                    <img id="imagePreview2" class="image-preview" style="display:none;">
                </div>
                
                <div class="form-group">
                    <label>URL Image 2</label>
                    <input type="text" id="styleUrl2" placeholder="Rempli automatiquement après upload" readonly style="opacity: 0.7;">
                    <small>Ce champ se remplit automatiquement après l'upload</small>
                </div>
                
                <div class="form-group">
                    <label>Descriptif <span class="required">*</span></label>
                    <textarea id="styleDescriptif" required placeholder="Description du style..."></textarea>
                </div>
                
                <div class="form-group">
                    <label>Prompt IA <span class="required">*</span></label>
                    <textarea id="stylePrompt" required style="min-height: 150px;" placeholder="Prompt pour l'IA..."></textarea>
                </div>
                
                <div class="form-group">
                    <label>Background <span class="required">*</span></label>
                    <div style="margin-top: 8px;">
                        <label class="background-option" onclick="selectBackground('auto')">
                            <input type="radio" name="background" id="backgroundAuto" value="auto" required>
                            <div class="background-option-label">
                                <div class="background-option-title">🎨 Fond automatique</div>
                                <div class="background-option-desc">L'IA choisit le meilleur fond (blanc, uni ou décor)</div>
                            </div>
                        </label>
                        <label class="background-option" onclick="selectBackground('prompt')">
                            <input type="radio" name="background" id="backgroundPrompt" value="prompt">
                            <div class="background-option-label">
                                <div class="background-option-title">📝 Background dans le prompt</div>
                                <div class="background-option-desc">Le fond est défini manuellement dans le prompt</div>
                            </div>
                        </label>
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Ordre d'affichage</label>
                    <input type="number" id="styleOrdre" min="0" max="9999" value="0" style="width: 100px;">
                    <small>Plus le nombre est petit, plus le style apparaît en premier</small>
                </div>
                
                <div class="form-group">
                    <label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">
                        <input type="checkbox" id="styleActif" checked style="width: auto; accent-color: var(--primary);">
                        <span style="text-transform: none; font-weight: 500;">Actif</span>
                    </label>
                </div>
                
                <div class="form-group">
                    <label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">
                        <input type="checkbox" id="styleFlagChampScene" style="width: auto; accent-color: var(--accent);">
                        <span style="text-transform: none; font-weight: 500;">🎬 Afficher le champ "Décrivez la scène"</span>
                    </label>
                    <small style="display: block; margin-top: 8px; color: var(--text-muted); margin-left: 30px;">
                        Si coché, l'utilisateur pourra décrire la scène pour aider l'IA
                    </small>
                </div>
                
                <div style="display: flex; gap: 1rem; margin-top: 2rem;">
                    <button type="submit" class="btn btn-primary">💾 Enregistrer</button>
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Annuler</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- MODAL CATEGORY -->
    <div id="categoryModal" class="modal">
        <div class="modal-content">
            <h3 id="categoryModalTitle">📁 Nouvelle Catégorie</h3>
            <form id="categoryForm" onsubmit="saveCategory(event)">
                <input type="hidden" id="categoryId">
                
                <div class="form-group">
                    <label>Slug (URL-friendly) <span class="required">*</span></label>
                    <input type="text" id="categorySlugInput" required placeholder="ex: affiche-vintage" 
                           pattern="[a-z0-9-]+" title="Minuscules, chiffres et tirets uniquement">
                    <small>Minuscules, chiffres et tirets uniquement</small>
                </div>
                
                <div class="form-group">
                    <label>Nom d'affichage <span class="required">*</span></label>
                    <input type="text" id="categoryNomInput" required placeholder="ex: Affiche Vintage">
                </div>
                
                <div class="form-group">
                    <label>Description</label>
                    <textarea id="categoryDescription" placeholder="Description de la catégorie..."></textarea>
                </div>
                
                <div class="form-group">
                    <label>Image de la catégorie</label>
                    <input type="file" id="categoryImageFile" accept="image/*" onchange="handleCategoryImageSelect(this)" 
                           style="margin-bottom: 10px; padding: 0.5rem; border-style: dashed;">
                    <input type="text" id="categoryImageUrl" placeholder="URL de l'image (auto-remplie)" readonly style="opacity: 0.7;">
                    <div id="categoryImagePreview" style="margin-top: 10px; display: none;">
                        <img id="categoryPreviewImg" src="" alt="Aperçu" style="max-width: 200px; max-height: 200px; border-radius: 12px; border: 2px solid var(--border);">
                    </div>
                    <small>Image affichée sur la page d'accueil (JPG, PNG, WebP, GIF - max 5MB)</small>
                </div>
                
                <div style="display: flex; gap: 1rem; margin-top: 2rem;">
                    <button type="submit" class="btn btn-primary">💾 Enregistrer</button>
                    <button type="button" class="btn btn-secondary" onclick="closeCategoryModal()">Annuler</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- IMAGE MODAL -->
    <div id="imageModal" onclick="closeImageModal()">
        <span class="close-image">&times;</span>
        <img id="imageModalContent" src="" alt="Style preview">
    </div>
    
    <script>
        const API = '<?php echo $_SERVER['PHP_SELF']; ?>';
        let allStyles = [];
        let allCategories = [];
        let uploadedImageUrl = null;
        let uploadedImageUrl2 = null;
        
        window.onload = async function() {
            await testConnection();
            await loadStats();
            await loadCategoriesForSelect();
            await loadStyles();
            await loadCategories();
            
            const uploadZone = document.getElementById('uploadZone');
            
            uploadZone.addEventListener('dragover', (e) => {
                e.preventDefault();
                uploadZone.classList.add('drag-over');
            });
            
            uploadZone.addEventListener('dragleave', () => {
                uploadZone.classList.remove('drag-over');
            });
            
            uploadZone.addEventListener('drop', (e) => {
                e.preventDefault();
                uploadZone.classList.remove('drag-over');
                
                const files = e.dataTransfer.files;
                if (files.length > 0) {
                    const input = document.getElementById('imageFile');
                    input.files = files;
                    handleImageSelect(input);
                }
            });
        };
        
        async function testConnection() {
            try {
                const res = await fetch(API + '?action=test');
                const data = await res.json();
                const status = document.getElementById('connectionStatus');
                
                if (data.success) {
                    status.className = 'connection-status status-ok';
                    status.textContent = '✅ Connexion PostgreSQL établie';
                } else {
                    status.className = 'connection-status status-error';
                    status.textContent = '❌ Erreur de connexion';
                }
            } catch(err) {
                const status = document.getElementById('connectionStatus');
                status.className = 'connection-status status-error';
                status.textContent = '❌ Impossible de se connecter: ' + err.message;
            }
        }
        
        function showTab(name) {
            document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
            document.querySelectorAll('.tab-content').forEach(t => t.classList.remove('active'));
            event.target.classList.add('active');
            document.getElementById('tab-' + name).classList.add('active');
            
            if (name === 'categories') {
                loadCategories();
            } else if (name === 'styles') {
                loadStyles();
            }
        }
        
        function showAlert(type, msg) {
            const el = type === 'success' ? 
                document.getElementById('alertSuccess') : 
                document.getElementById('alertError');
            el.textContent = msg;
            el.classList.add('active');
            setTimeout(() => el.classList.remove('active'), 5000);
        }
        
        function selectBackground(value) {
            document.querySelectorAll('.background-option').forEach(opt => {
                opt.classList.remove('selected');
            });
            event.currentTarget.classList.add('selected');
        }
        
        async function loadCategoriesForSelect() {
            try {
                const res = await fetch(API + '?action=list_categories');
                const data = await res.json();
                
                if (data.success) {
                    allCategories = data.data;
                    populateCategorySelect();
                }
            } catch(err) {
                console.error('Erreur chargement catégories:', err);
            }
        }
        
        function populateCategorySelect() {
            const select = document.getElementById('styleCategorie');
            select.innerHTML = '<option value="">-- Sélectionner une catégorie --</option>';
            
            allCategories.forEach(cat => {
                const option = document.createElement('option');
                option.value = cat.slug;
                option.textContent = cat.nom;
                option.dataset.nom = cat.nom;
                select.appendChild(option);
            });
        }
        
        async function loadStats() {
            try {
                const res = await fetch(API + '?action=stats');
                const data = await res.json();
                if (data.success) {
                    document.getElementById('statTotal').textContent = data.stats.total;
                    document.getElementById('statActifs').textContent = data.stats.actifs;
                    document.getElementById('statCategories').textContent = data.stats.categories;
                }
            } catch(err) {
                showAlert('error', 'Erreur chargement stats: ' + err.message);
            }
        }
        
        async function loadStyles() {
            try {
                const res = await fetch(API + '?action=list');
                const data = await res.json();
                
                if (data.success) {
                    allStyles = data.data;
                    renderStylesTable(allStyles);
                    restoreFilters();
                } else {
                    throw new Error(data.error || 'Erreur inconnue');
                }
            } catch(err) {
                document.getElementById('stylesTable').innerHTML = 
                    `<div style="padding: 3rem; text-align: center; color: var(--danger);">❌ ${err.message}</div>`;
                showAlert('error', 'Erreur chargement styles: ' + err.message);
            }
        }
        
        function renderStylesTable(styles) {
            const categories = [...new Set(allStyles.map(s => s.categorie_nom))].sort();
            
            let html = `
                <table>
                    <thead>
                        <tr>
                            <th>
                                <div class="header-content">
                                    <span class="header-label">ID</span>
                                    <div class="header-filter">
                                        <input type="number" id="filterID" placeholder="Filtrer..." onkeyup="filterTableByHeaders()">
                                    </div>
                                </div>
                            </th>
                            <th>
                                <div class="header-content">
                                    <span class="header-label">Aperçu</span>
                                    <div class="no-filter"></div>
                                </div>
                            </th>
                            <th>
                                <div class="header-content">
                                    <span class="header-label">Titre</span>
                                    <div class="header-filter">
                                        <input type="text" id="filterTitre" placeholder="Filtrer..." onkeyup="filterTableByHeaders()">
                                    </div>
                                </div>
                            </th>
                            <th>
                                <div class="header-content">
                                    <span class="header-label">Catégorie</span>
                                    <div class="header-filter">
                                        <select id="filterCategorie" onchange="filterTableByHeaders()">
                                            <option value="">Toutes</option>
                                            ${categories.map(cat => `<option value="${cat}">${cat}</option>`).join('')}
                                        </select>
                                    </div>
                                </div>
                            </th>
                            <th>
                                <div class="header-content">
                                    <span class="header-label">Statut</span>
                                    <div class="header-filter">
                                        <select id="filterStatut" onchange="filterTableByHeaders()">
                                            <option value="">Tous</option>
                                            <option value="actif">Actif</option>
                                            <option value="inactif">Inactif</option>
                                        </select>
                                    </div>
                                </div>
                            </th>
                            <th style="text-align: right;">
                                <div class="header-content">
                                    <span class="header-label">Actions</span>
                                    <div class="no-filter"></div>
                                </div>
                            </th>
                        </tr>
                    </thead>
                    <tbody>
            `;
            
            if (styles.length === 0) {
                html += '<tr><td colspan="6" style="text-align:center; padding: 3rem; color: var(--text-muted);">Aucun style trouvé</td></tr>';
            } else {
                styles.forEach(s => {
                    const statusBadge = s.actif ? 
                        '<span class="badge badge-active">Actif</span>' : 
                        '<span class="badge badge-inactive">Inactif</span>';
                    
                    let imageUrl = s.url;
                    if (!imageUrl || imageUrl === '') {
                        const imageTitle = s.titre.replace(/[^a-zA-Z0-9_-]/g, '_').replace(/_+/g, '_').replace(/^_|_$/g, '');
                        imageUrl = `/styles/${imageTitle}.png`;
                    }
                    
                    html += `
                        <tr>
                            <td><strong style="color: var(--primary);">${s.style_id}</strong></td>
                            <td>
                                <img src="${imageUrl}" 
                                     alt="${s.titre}" 
                                     class="style-thumbnail" 
                                     onclick="openImageModal('${imageUrl}', '${s.titre}')"
                                     onerror="this.classList.add('error'); this.innerHTML='🖼️'; this.removeAttribute('src');">
                            </td>
                            <td>${s.titre}</td>
                            <td><code>${s.categorie_nom}</code></td>
                            <td>${statusBadge}</td>
                            <td class="actions" style="justify-content: flex-end;">
                                <button class="btn ${s.actif ? 'btn-warning' : 'btn-success'} btn-sm" 
                                        onclick="toggleStyle(${s.id}, ${s.actif})" 
                                        title="${s.actif ? 'Désactiver' : 'Activer'}">
                                    ${s.actif ? '👁️' : '✓'}
                                </button>
                                <button class="btn btn-primary btn-sm" onclick="editStyleById(${s.id})">✏️</button>
                                <button class="btn btn-danger btn-sm" onclick="deleteStyle(${s.id})">🗑️</button>
                            </td>
                        </tr>
                    `;
                });
            }
            
            html += '</tbody></table>';
            document.getElementById('stylesTable').innerHTML = html;
        }
        
        async function openModal() {
            document.getElementById('styleForm').reset();
            document.getElementById('styleDbId').value = '';
            document.getElementById('modalTitle').textContent = '✨ Nouveau Style';
            document.getElementById('styleActif').checked = true;
            document.getElementById('styleOrdre').value = 0;
            uploadedImageUrl = null;
            uploadedImageUrl2 = null;
            
            document.getElementById('imagePreview').style.display = 'none';
            document.getElementById('uploadProgress').classList.remove('active');
            document.querySelectorAll('.background-option').forEach(opt => opt.classList.remove('selected'));
            
            populateCategorySelect();
            
            try {
                const res = await fetch(API + '?action=get_next_id');
                const data = await res.json();
                if (data.success) {
                    document.getElementById('styleId').value = data.next_id;
                }
            } catch(err) {
                document.getElementById('styleId').value = 1;
            }
            
            document.getElementById('modal').classList.add('active');
        }
        
        function closeModal() {
            document.getElementById('modal').classList.remove('active');
        }
        
        function editStyleById(id) {
            const style = allStyles.find(s => s.id === id);
            if (!style) {
                showAlert('error', 'Style introuvable');
                return;
            }
            editStyle(style);
        }
        
        function editStyle(style) {
            document.getElementById('styleDbId').value = style.id;
            document.getElementById('styleId').value = style.style_id;
            document.getElementById('styleTitre').value = style.titre;
            
            populateCategorySelect();
            document.getElementById('styleCategorie').value = style.categorie_slug;
            document.getElementById('categorieSlug').value = style.categorie_slug;
            document.getElementById('categorieNom').value = style.categorie_nom;
            
            document.getElementById('styleDescriptif').value = style.descriptif;
            document.getElementById('stylePrompt').value = style.prompt;
            
            let backgroundValue = style.background || 'auto';
            if (['fond_uni', 'fond_colore', 'fond_ia'].includes(backgroundValue)) {
                backgroundValue = 'auto';
            } else if (backgroundValue === 'fond_prompt') {
                backgroundValue = 'prompt';
            }
            
            const backgroundRadio = document.querySelector(`input[name="background"][value="${backgroundValue}"]`);
            if (backgroundRadio) {
                backgroundRadio.checked = true;
                backgroundRadio.closest('.background-option').classList.add('selected');
            }
            
            document.getElementById('styleUrl').value = style.url || '';
            document.getElementById('styleUrl2').value = style.url2 || '';
            
            if (style.url) {
                const preview = document.getElementById('imagePreview');
                preview.src = style.url;
                preview.style.display = 'block';
                uploadedImageUrl = style.url;
            }
            
            if (style.url2) {
                const preview2 = document.getElementById('imagePreview2');
                preview2.src = style.url2;
                preview2.style.display = 'block';
                uploadedImageUrl2 = style.url2;
            }
            
            document.getElementById('styleActif').checked = style.actif;
            document.getElementById('styleOrdre').value = style.ordre || 0;
            document.getElementById('styleFlagChampScene').checked = style.flag_champ_scene || false;
            document.getElementById('modalTitle').textContent = '✏️ Modifier Style #' + style.style_id;
            document.getElementById('modal').classList.add('active');
        }
        
        async function handleImageSelect(input) {
            const file = input.files[0];
            if (!file) return;
            
            if (file.size > 5242880) {
                showAlert('error', 'Le fichier est trop volumineux (max 5 MB)');
                input.value = '';
                return;
            }
            
            const reader = new FileReader();
            reader.onload = (e) => {
                const preview = document.getElementById('imagePreview');
                preview.src = e.target.result;
                preview.style.display = 'block';
            };
            reader.readAsDataURL(file);
            
            const formData = new FormData();
            formData.append('image', file);
            
            const titre = document.getElementById('styleTitre').value || '';
            formData.append('titre', titre);
            
            const progress = document.getElementById('uploadProgress');
            const progressFill = document.getElementById('progressFill');
            progress.classList.add('active');
            
            try {
                const xhr = new XMLHttpRequest();
                
                xhr.upload.addEventListener('progress', (e) => {
                    if (e.lengthComputable) {
                        const percent = (e.loaded / e.total) * 100;
                        progressFill.style.width = percent + '%';
                    }
                });
                
                xhr.addEventListener('load', () => {
                    if (xhr.status === 200) {
                        const data = JSON.parse(xhr.responseText);
                        if (data.success) {
                            uploadedImageUrl = data.url;
                            const urlField = document.getElementById('styleUrl');
                            urlField.value = data.url;
                            showAlert('success', '✅ Image uploadée : ' + data.url);
                        } else {
                            showAlert('error', data.error || 'Erreur upload');
                        }
                    } else {
                        showAlert('error', 'Erreur serveur: ' + xhr.status);
                    }
                    progress.classList.remove('active');
                    progressFill.style.width = '0';
                });
                
                xhr.addEventListener('error', () => {
                    showAlert('error', 'Erreur réseau lors de l\'upload');
                    progress.classList.remove('active');
                    progressFill.style.width = '0';
                });
                
                xhr.open('POST', API + '?action=upload_image');
                xhr.send(formData);
                
            } catch(err) {
                showAlert('error', 'Erreur upload: ' + err.message);
                progress.classList.remove('active');
                progressFill.style.width = '0';
            }
        }
        
        async function handleImageSelect2(input) {
            const file = input.files[0];
            if (!file) return;
            
            if (file.size > 5242880) {
                showAlert('error', 'Le fichier est trop volumineux (max 5 MB)');
                input.value = '';
                return;
            }
            
            const reader = new FileReader();
            reader.onload = (e) => {
                const preview = document.getElementById('imagePreview2');
                preview.src = e.target.result;
                preview.style.display = 'block';
            };
            reader.readAsDataURL(file);
            
            const formData = new FormData();
            formData.append('image', file);
            
            const titre = document.getElementById('styleTitre').value || '';
            formData.append('titre', titre + '_2');
            
            const progress = document.getElementById('uploadProgress2');
            const progressFill = document.getElementById('progressFill2');
            progress.classList.add('active');
            
            try {
                const xhr = new XMLHttpRequest();
                
                xhr.upload.addEventListener('progress', (e) => {
                    if (e.lengthComputable) {
                        const percent = (e.loaded / e.total) * 100;
                        progressFill.style.width = percent + '%';
                    }
                });
                
                xhr.addEventListener('load', () => {
                    if (xhr.status === 200) {
                        const data = JSON.parse(xhr.responseText);
                        if (data.success) {
                            uploadedImageUrl2 = data.url;
                            const urlField = document.getElementById('styleUrl2');
                            urlField.value = data.url;
                            showAlert('success', '✅ Image 2 uploadée : ' + data.url);
                        } else {
                            showAlert('error', data.error || 'Erreur upload');
                        }
                    } else {
                        showAlert('error', 'Erreur serveur: ' + xhr.status);
                    }
                    progress.classList.remove('active');
                    progressFill.style.width = '0';
                });
                
                xhr.addEventListener('error', () => {
                    showAlert('error', 'Erreur réseau lors de l\'upload');
                    progress.classList.remove('active');
                    progressFill.style.width = '0';
                });
                
                xhr.open('POST', API + '?action=upload_image');
                xhr.send(formData);
                
            } catch(err) {
                showAlert('error', 'Erreur upload: ' + err.message);
                progress.classList.remove('active');
                progressFill.style.width = '0';
            }
        }
        
        async function handleCategoryImageSelect(input) {
            if (!input.files || !input.files[0]) return;
            
            const file = input.files[0];
            
            if (file.size > 5 * 1024 * 1024) {
                showAlert('error', 'Image trop volumineuse (max 5 MB)');
                return;
            }
            
            const preview = document.getElementById('categoryImagePreview');
            const img = document.getElementById('categoryPreviewImg');
            
            const reader = new FileReader();
            reader.onload = function(e) {
                img.src = e.target.result;
                preview.style.display = 'block';
            };
            reader.readAsDataURL(file);
            
            const formData = new FormData();
            formData.append('image', file);
            
            const categorySlug = document.getElementById('categorySlugInput').value || 'temp';
            formData.append('category_slug', categorySlug);
            
            try {
                const res = await fetch(API + '?action=upload_category_image', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await res.json();
                
                if (data.success) {
                    document.getElementById('categoryImageUrl').value = data.url;
                    showAlert('success', '✅ Image uploadée : ' + data.url);
                } else {
                    showAlert('error', data.error || 'Erreur upload');
                }
            } catch(err) {
                showAlert('error', 'Erreur upload: ' + err.message);
            }
        }
        
        async function saveStyle(e) {
            e.preventDefault();
            
            const id = document.getElementById('styleDbId').value;
            const backgroundRadio = document.querySelector('input[name="background"]:checked');
            
            // Debug: vérifier si l'élément existe
            const flagElement = document.getElementById('styleFlagChampScene');
            console.log('🔍 Element flag_champ_scene:', flagElement);
            console.log('🔍 Valeur checked:', flagElement ? flagElement.checked : 'ELEMENT NOT FOUND');
            
            const data = {
                id: id || null,
                style_id: parseInt(document.getElementById('styleId').value),
                titre: document.getElementById('styleTitre').value,
                theme: 'Personnage',
                categorie_slug: document.getElementById('categorieSlug').value,
                categorie_nom: document.getElementById('categorieNom').value,
                descriptif: document.getElementById('styleDescriptif').value,
                prompt: document.getElementById('stylePrompt').value,
                background: backgroundRadio ? backgroundRadio.value : 'auto',
                url: uploadedImageUrl || document.getElementById('styleUrl').value,
                url2: uploadedImageUrl2 || document.getElementById('styleUrl2').value,
                ordre: parseInt(document.getElementById('styleOrdre').value) || 0,
                actif: document.getElementById('styleActif').checked,
                flag_champ_scene: flagElement ? flagElement.checked : false
            };
            
            console.log('📤 Data envoyée:', data);
            
            const action = id ? 'update' : 'create';
            
            try {
                const res = await fetch(API + '?action=' + action, {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(data)
                });
                
                const result = await res.json();
                
                if (result.success) {
                    showAlert('success', '✅ Style enregistré avec succès !');
                    closeModal();
                    await loadStyles();
                    await loadStats();
                } else {
                    showAlert('error', result.error || 'Erreur inconnue');
                }
            } catch(err) {
                showAlert('error', 'Erreur réseau: ' + err.message);
            }
        }
        
        async function deleteStyle(id) {
            if (!confirm('⚠️ Supprimer définitivement ce style ?')) return;
            
            try {
                const res = await fetch(API + '?action=delete&id=' + id);
                const result = await res.json();
                
                if (result.success) {
                    showAlert('success', '✅ Style supprimé');
                    await loadStyles();
                    await loadStats();
                } else {
                    showAlert('error', result.error || 'Erreur suppression');
                }
            } catch(err) {
                showAlert('error', 'Erreur réseau: ' + err.message);
            }
        }
        
        async function toggleStyle(id, currentStatus) {
            try {
                const res = await fetch(API + '?action=toggle&id=' + id, { method: 'POST' });
                const result = await res.json();
                
                if (result.success) {
                    const newStatus = result.actif ? 'activé' : 'désactivé';
                    showAlert('success', `✅ Style ${newStatus}`);
                    await loadStyles();
                    await loadStats();
                } else {
                    showAlert('error', result.error || 'Erreur toggle');
                }
            } catch(err) {
                showAlert('error', 'Erreur réseau: ' + err.message);
            }
        }
        
        function filterTableByHeaders() {
            const filterID = document.getElementById('filterID')?.value || '';
            const filterTitre = document.getElementById('filterTitre')?.value.toLowerCase() || '';
            const filterCategorie = document.getElementById('filterCategorie')?.value || '';
            const filterStatut = document.getElementById('filterStatut')?.value || '';
            
            localStorage.setItem('filter_id', filterID);
            localStorage.setItem('filter_titre', filterTitre);
            localStorage.setItem('filter_categorie', filterCategorie);
            localStorage.setItem('filter_statut', filterStatut);
            
            const filtered = allStyles.filter(s => {
                const matchID = !filterID || s.style_id.toString().includes(filterID);
                const matchTitre = !filterTitre || s.titre.toLowerCase().includes(filterTitre);
                const matchCategorie = !filterCategorie || s.categorie_nom === filterCategorie;
                const matchStatut = !filterStatut || 
                    (filterStatut === 'actif' && s.actif) || 
                    (filterStatut === 'inactif' && !s.actif);
                
                return matchID && matchTitre && matchCategorie && matchStatut;
            });
            
            renderStylesTable(filtered);
        }
        
        function restoreFilters() {
            const filterID = localStorage.getItem('filter_id') || '';
            const filterTitre = localStorage.getItem('filter_titre') || '';
            const filterCategorie = localStorage.getItem('filter_categorie') || '';
            const filterStatut = localStorage.getItem('filter_statut') || '';
            
            setTimeout(() => {
                if (document.getElementById('filterID')) document.getElementById('filterID').value = filterID;
                if (document.getElementById('filterTitre')) document.getElementById('filterTitre').value = filterTitre;
                if (document.getElementById('filterCategorie')) document.getElementById('filterCategorie').value = filterCategorie;
                if (document.getElementById('filterStatut')) document.getElementById('filterStatut').value = filterStatut;
                
                if (filterID || filterTitre || filterCategorie || filterStatut) {
                    filterTableByHeaders();
                }
            }, 100);
        }
        
        function clearFilters() {
            localStorage.removeItem('filter_id');
            localStorage.removeItem('filter_titre');
            localStorage.removeItem('filter_categorie');
            localStorage.removeItem('filter_statut');
            
            if (document.getElementById('filterID')) document.getElementById('filterID').value = '';
            if (document.getElementById('filterTitre')) document.getElementById('filterTitre').value = '';
            if (document.getElementById('filterCategorie')) document.getElementById('filterCategorie').value = '';
            if (document.getElementById('filterStatut')) document.getElementById('filterStatut').value = '';
            
            renderStylesTable(allStyles);
        }
        
        async function loadCategories() {
            try {
                const res = await fetch(API + '?action=list_categories');
                const data = await res.json();
                
                if (data.success) {
                    allCategories = data.data;
                    renderCategoriesTable(data.data);
                } else {
                    showAlert('error', 'Erreur: ' + (data.error || 'Inconnue'));
                }
            } catch(err) {
                showAlert('error', 'Erreur chargement catégories: ' + err.message);
            }
        }
        
        function renderCategoriesTable(categories) {
            let html = `
                <table>
                    <thead>
                        <tr>
                            <th>Image</th>
                            <th>Slug</th>
                            <th>Nom</th>
                            <th>Statut</th>
                            <th>Styles</th>
                            <th style="text-align: right;">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
            `;
            
            if (categories.length === 0) {
                html += '<tr><td colspan="6" style="text-align:center; padding: 3rem; color: var(--text-muted);">Aucune catégorie trouvée</td></tr>';
            } else {
                categories.forEach(cat => {
                    const statusBadge = cat.actif ? 
                        '<span class="badge badge-active">Actif</span>' : 
                        '<span class="badge badge-inactive">Inactif</span>';
                    
                    const imageHtml = cat.url_img_categorie ? 
                        `<img src="${cat.url_img_categorie}" alt="${cat.nom}" class="style-thumbnail" onclick="openImageModal('${cat.url_img_categorie}', '${cat.nom}')">` :
                        '<span style="color: var(--text-subtle);">—</span>';
                    
                    html += `
                        <tr>
                            <td>${imageHtml}</td>
                            <td><code>${cat.slug}</code></td>
                            <td><strong>${cat.nom}</strong></td>
                            <td>${statusBadge}</td>
                            <td>
                                ${cat.nb_styles || 0} total
                                ${cat.nb_styles_actifs !== undefined ? 
                                    `<span style="color: var(--primary); margin-left: 8px;">(${cat.nb_styles_actifs} actifs)</span>` 
                                    : ''}
                            </td>
                            <td class="actions" style="justify-content: flex-end;">
                                <button class="btn ${cat.actif ? 'btn-warning' : 'btn-success'} btn-sm" 
                                        onclick="toggleCategory('${cat.slug}', ${cat.actif})" 
                                        title="${cat.actif ? 'Désactiver' : 'Activer'}">
                                    ${cat.actif ? '👁️' : '✓'}
                                </button>
                                <button class="btn btn-primary btn-sm" onclick="editCategoryBySlug('${cat.slug}')">✏️</button>
                            </td>
                        </tr>
                    `;
                });
            }
            
            html += '</tbody></table>';
            document.getElementById('categoriesTable').innerHTML = html;
        }
        
        function openCategoryModal() {
            document.getElementById('categoryForm').reset();
            document.getElementById('categoryId').value = '';
            document.getElementById('categoryImageUrl').value = '';
            document.getElementById('categoryImagePreview').style.display = 'none';
            document.getElementById('categorySlugInput').readOnly = false;
            document.getElementById('categoryModalTitle').textContent = '📁 Nouvelle Catégorie';
            document.getElementById('categoryModal').classList.add('active');
        }
        
        function closeCategoryModal() {
            document.getElementById('categoryModal').classList.remove('active');
            document.getElementById('categorySlugInput').readOnly = false;
            document.getElementById('categoryImagePreview').style.display = 'none';
        }
        
        async function toggleCategory(slug, currentStatus) {
            try {
                const res = await fetch(API + '?action=toggle_category&slug=' + encodeURIComponent(slug), { method: 'POST' });
                const result = await res.json();
                
                if (result.success) {
                    const newStatus = result.actif ? 'activée' : 'désactivée';
                    showAlert('success', `✅ Catégorie ${newStatus}`);
                    await loadCategories();
                    await loadStyles();
                    await loadStats();
                } else {
                    showAlert('error', result.error || 'Erreur toggle catégorie');
                }
            } catch(err) {
                showAlert('error', 'Erreur réseau: ' + err.message);
            }
        }
        
        function editCategoryBySlug(slug) {
            const category = allCategories.find(c => c.slug === slug);
            if (!category) {
                showAlert('error', 'Catégorie introuvable');
                return;
            }
            editCategory(category);
        }
        
        function editCategory(category) {
            document.getElementById('categoryId').value = category.slug;
            document.getElementById('categorySlugInput').value = category.slug;
            document.getElementById('categorySlugInput').readOnly = true;
            document.getElementById('categoryNomInput').value = category.nom;
            document.getElementById('categoryDescription').value = category.description || '';
            document.getElementById('categoryImageUrl').value = category.url_img_categorie || '';
            
            const preview = document.getElementById('categoryImagePreview');
            const img = document.getElementById('categoryPreviewImg');
            if (category.url_img_categorie) {
                img.src = category.url_img_categorie;
                preview.style.display = 'block';
            } else {
                preview.style.display = 'none';
            }
            
            document.getElementById('categoryModalTitle').textContent = '✏️ Modifier Catégorie';
            document.getElementById('categoryModal').classList.add('active');
        }
        
        async function saveCategory(e) {
            e.preventDefault();
            
            const oldSlug = document.getElementById('categoryId').value;
            const newSlug = document.getElementById('categorySlugInput').value;
            const newNom = document.getElementById('categoryNomInput').value;
            const description = document.getElementById('categoryDescription').value;
            const urlImgCategorie = document.getElementById('categoryImageUrl').value;
            
            try {
                if (oldSlug) {
                    const res = await fetch(API + '?action=update_category', {
                        method: 'POST',
                        headers: {'Content-Type': 'application/json'},
                        body: JSON.stringify({
                            old_slug: oldSlug,
                            new_nom: newNom,
                            url_img_categorie: urlImgCategorie
                        })
                    });
                    
                    const result = await res.json();
                    
                    if (result.success) {
                        showAlert('success', '✅ Catégorie mise à jour !');
                        closeCategoryModal();
                        await loadCategoriesForSelect();
                        await loadStyles();
                        await loadCategories();
                    } else {
                        showAlert('error', result.error || 'Erreur mise à jour');
                    }
                } else {
                    const res = await fetch(API + '?action=create_category', {
                        method: 'POST',
                        headers: {'Content-Type': 'application/json'},
                        body: JSON.stringify({
                            slug: newSlug,
                            nom: newNom,
                            description: description,
                            url_img_categorie: urlImgCategorie,
                            ordre: 0
                        })
                    });
                    
                    const result = await res.json();
                    
                    if (result.success) {
                        showAlert('success', '✅ Catégorie créée avec succès !');
                        closeCategoryModal();
                        await loadCategoriesForSelect();
                        await loadCategories();
                    } else {
                        showAlert('error', result.error || 'Erreur création');
                    }
                }
            } catch(err) {
                showAlert('error', 'Erreur réseau: ' + err.message);
            }
        }
        
        async function importJSON(input) {
            if (!input.files[0]) return;
            
            if (!confirm('⚠️ Cette action va REMPLACER toutes les données existantes. Continuer ?')) {
                input.value = '';
                return;
            }
            
            const file = input.files[0];
            const reader = new FileReader();
            
            reader.onload = async function(e) {
                try {
                    const json = JSON.parse(e.target.result);
                    
                    const res = await fetch(API + '?action=import', {
                        method: 'POST',
                        headers: {'Content-Type': 'application/json'},
                        body: JSON.stringify(json)
                    });
                    
                    const result = await res.json();
                    
                    if (result.success) {
                        showAlert('success', result.message);
                        await loadStyles();
                        await loadStats();
                    } else {
                        showAlert('error', result.error || 'Erreur import');
                    }
                } catch (err) {
                    showAlert('error', 'Erreur JSON: ' + err.message);
                } finally {
                    input.value = '';
                }
            };
            
            reader.readAsText(file);
        }
        
        async function exportJSON() {
            try {
                const res = await fetch(API + '?action=export');
                const data = await res.text();
                
                const blob = new Blob([data], {type: 'application/json'});
                const url = URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `styles_export_${new Date().toISOString().split('T')[0]}.json`;
                a.click();
                URL.revokeObjectURL(url);
                
                showAlert('success', '✅ Export téléchargé !');
            } catch(err) {
                showAlert('error', 'Erreur export: ' + err.message);
            }
        }
        
        // Modal se ferme uniquement via bouton Annuler ou touche Escape
        // Clic sur le fond désactivé pour éviter les fermetures accidentelles
        document.getElementById('modal').addEventListener('click', function(e) {
            // Ne rien faire - fermeture uniquement via bouton ou Escape
        });
        
        document.getElementById('categoryModal').addEventListener('click', function(e) {
            // Ne rien faire - fermeture uniquement via bouton ou Escape
        });
        
        document.getElementById('styleCategorie').addEventListener('change', function(e) {
            const selectedOption = e.target.options[e.target.selectedIndex];
            const slug = selectedOption.value;
            const nom = selectedOption.dataset.nom || selectedOption.textContent;
            
            document.getElementById('categorieSlug').value = slug;
            document.getElementById('categorieNom').value = nom;
        });
        
        function openImageModal(imageUrl, title) {
            const modal = document.getElementById('imageModal');
            const img = document.getElementById('imageModalContent');
            img.src = imageUrl;
            img.alt = title;
            modal.classList.add('active');
        }
        
        function closeImageModal() {
            document.getElementById('imageModal').classList.remove('active');
        }
        
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeImageModal();
                closeModal();
                closeCategoryModal();
            }
        });
        
        // ============================================
        // GESTION DE L'ORDRE
        // ============================================
        let ordreStyles = [];
        
        function populateOrdreCategories() {
            const select = document.getElementById('ordreFilterCategory');
            if (!select) return;
            
            select.innerHTML = '<option value="all">Toutes les catégories</option>';
            
            const categories = [...new Set(allStyles.map(s => s.categorie_slug))];
            categories.forEach(slug => {
                const style = allStyles.find(s => s.categorie_slug === slug);
                const nom = style ? style.categorie_nom : slug;
                select.innerHTML += `<option value="${slug}">${nom}</option>`;
            });
        }
        
        async function loadOrdreList() {
            const container = document.getElementById('ordreList');
            const categoryFilter = document.getElementById('ordreFilterCategory')?.value || 'all';
            const activeOnly = document.getElementById('ordreShowActiveOnly')?.checked ?? true;
            
            // Filtrer les styles
            ordreStyles = allStyles.filter(s => {
                if (categoryFilter !== 'all' && s.categorie_slug !== categoryFilter) return false;
                if (activeOnly && !s.actif) return false;
                return true;
            });
            
            // Trier par ordre actuel
            ordreStyles.sort((a, b) => (a.ordre || 0) - (b.ordre || 0));
            
            if (ordreStyles.length === 0) {
                container.innerHTML = '<p style="color: var(--text-muted); padding: 2rem; text-align: center;">Aucun style à afficher</p>';
                return;
            }
            
            container.innerHTML = ordreStyles.map((style, index) => `
                <div class="ordre-item" data-id="${style.id}" data-index="${index}" style="
                    display: flex;
                    align-items: center;
                    gap: 12px;
                    padding: 12px 16px;
                    background: var(--bg-dark);
                    border: 1px solid var(--border);
                    border-radius: 8px;
                    transition: all 0.2s;
                ">
                    <div style="display: flex; flex-direction: column; gap: 4px;">
                        <button onclick="moveStyleUp(${index})" class="btn-move" ${index === 0 ? 'disabled' : ''} style="
                            padding: 4px 8px;
                            background: ${index === 0 ? 'var(--bg-darker)' : 'var(--primary)'};
                            border: none;
                            border-radius: 4px;
                            color: white;
                            cursor: ${index === 0 ? 'not-allowed' : 'pointer'};
                            font-size: 12px;
                        ">⬆️</button>
                        <button onclick="moveStyleDown(${index})" class="btn-move" ${index === ordreStyles.length - 1 ? 'disabled' : ''} style="
                            padding: 4px 8px;
                            background: ${index === ordreStyles.length - 1 ? 'var(--bg-darker)' : 'var(--primary)'};
                            border: none;
                            border-radius: 4px;
                            color: white;
                            cursor: ${index === ordreStyles.length - 1 ? 'not-allowed' : 'pointer'};
                            font-size: 12px;
                        ">⬇️</button>
                    </div>
                    <span style="
                        background: var(--primary);
                        color: white;
                        padding: 4px 10px;
                        border-radius: 20px;
                        font-size: 0.85rem;
                        font-weight: 600;
                        min-width: 40px;
                        text-align: center;
                    ">${index + 1}</span>
                    <img src="${style.url || '/styles/default.png'}" alt="${style.titre}" style="
                        width: 50px;
                        height: 50px;
                        object-fit: cover;
                        border-radius: 8px;
                        border: 1px solid var(--border);
                    " onerror="this.src='data:image/svg+xml,<svg xmlns=%22http://www.w3.org/2000/svg%22 viewBox=%220 0 100 100%22><rect fill=%22%23333%22 width=%22100%22 height=%22100%22/><text x=%2250%22 y=%2250%22 text-anchor=%22middle%22 dy=%22.3em%22 fill=%22%23666%22 font-size=%2240%22>?</text></svg>'">
                    <div style="flex: 1;">
                        <div style="font-weight: 600; color: var(--text-light);">${style.titre}</div>
                        <div style="font-size: 0.85rem; color: var(--text-muted);">${style.categorie_nom} • ID: ${style.style_id}</div>
                    </div>
                    <span style="
                        padding: 4px 10px;
                        border-radius: 12px;
                        font-size: 0.8rem;
                        background: ${style.actif ? 'rgba(74, 222, 128, 0.2)' : 'rgba(239, 68, 68, 0.2)'};
                        color: ${style.actif ? '#4ade80' : '#ef4444'};
                    ">${style.actif ? 'Actif' : 'Inactif'}</span>
                    <span style="color: var(--text-muted); font-size: 0.85rem;">Ordre: ${style.ordre || 0}</span>
                </div>
            `).join('');
        }
        
        function moveStyleUp(index) {
            if (index <= 0) return;
            
            const temp = ordreStyles[index];
            ordreStyles[index] = ordreStyles[index - 1];
            ordreStyles[index - 1] = temp;
            
            renderOrdreList();
        }
        
        function moveStyleDown(index) {
            if (index >= ordreStyles.length - 1) return;
            
            const temp = ordreStyles[index];
            ordreStyles[index] = ordreStyles[index + 1];
            ordreStyles[index + 1] = temp;
            
            renderOrdreList();
        }
        
        function renderOrdreList() {
            const container = document.getElementById('ordreList');
            
            container.innerHTML = ordreStyles.map((style, index) => `
                <div class="ordre-item" data-id="${style.id}" data-index="${index}" style="
                    display: flex;
                    align-items: center;
                    gap: 12px;
                    padding: 12px 16px;
                    background: var(--bg-dark);
                    border: 1px solid var(--border);
                    border-radius: 8px;
                    transition: all 0.2s;
                ">
                    <div style="display: flex; flex-direction: column; gap: 4px;">
                        <button onclick="moveStyleUp(${index})" class="btn-move" ${index === 0 ? 'disabled' : ''} style="
                            padding: 4px 8px;
                            background: ${index === 0 ? 'var(--bg-darker)' : 'var(--primary)'};
                            border: none;
                            border-radius: 4px;
                            color: white;
                            cursor: ${index === 0 ? 'not-allowed' : 'pointer'};
                            font-size: 12px;
                        ">⬆️</button>
                        <button onclick="moveStyleDown(${index})" class="btn-move" ${index === ordreStyles.length - 1 ? 'disabled' : ''} style="
                            padding: 4px 8px;
                            background: ${index === ordreStyles.length - 1 ? 'var(--bg-darker)' : 'var(--primary)'};
                            border: none;
                            border-radius: 4px;
                            color: white;
                            cursor: ${index === ordreStyles.length - 1 ? 'not-allowed' : 'pointer'};
                            font-size: 12px;
                        ">⬇️</button>
                    </div>
                    <span style="
                        background: var(--primary);
                        color: white;
                        padding: 4px 10px;
                        border-radius: 20px;
                        font-size: 0.85rem;
                        font-weight: 600;
                        min-width: 40px;
                        text-align: center;
                    ">${index + 1}</span>
                    <img src="${style.url || '/styles/default.png'}" alt="${style.titre}" style="
                        width: 50px;
                        height: 50px;
                        object-fit: cover;
                        border-radius: 8px;
                        border: 1px solid var(--border);
                    " onerror="this.src='data:image/svg+xml,<svg xmlns=%22http://www.w3.org/2000/svg%22 viewBox=%220 0 100 100%22><rect fill=%22%23333%22 width=%22100%22 height=%22100%22/><text x=%2250%22 y=%2250%22 text-anchor=%22middle%22 dy=%22.3em%22 fill=%22%23666%22 font-size=%2240%22>?</text></svg>'">
                    <div style="flex: 1;">
                        <div style="font-weight: 600; color: var(--text-light);">${style.titre}</div>
                        <div style="font-size: 0.85rem; color: var(--text-muted);">${style.categorie_nom} • ID: ${style.style_id}</div>
                    </div>
                    <span style="
                        padding: 4px 10px;
                        border-radius: 12px;
                        font-size: 0.8rem;
                        background: ${style.actif ? 'rgba(74, 222, 128, 0.2)' : 'rgba(239, 68, 68, 0.2)'};
                        color: ${style.actif ? '#4ade80' : '#ef4444'};
                    ">${style.actif ? 'Actif' : 'Inactif'}</span>
                    <span style="color: var(--text-muted); font-size: 0.85rem;">Ordre: ${index + 1}</span>
                </div>
            `).join('');
        }
        
        async function saveAllOrders() {
            if (ordreStyles.length === 0) {
                showAlert('error', 'Aucun style à sauvegarder');
                return;
            }
            
            const orders = ordreStyles.map((style, index) => ({
                id: style.id,
                ordre: index + 1
            }));
            
            try {
                const res = await fetch(API + '?action=save_order', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({ orders })
                });
                
                const result = await res.json();
                
                if (result.success) {
                    showAlert('success', '✅ Ordre sauvegardé pour ' + result.count + ' styles !');
                    // Recharger les styles pour mettre à jour les ordres
                    await loadStyles();
                    loadOrdreList();
                } else {
                    showAlert('error', result.error || 'Erreur sauvegarde');
                }
            } catch(err) {
                showAlert('error', 'Erreur: ' + err.message);
            }
        }
        
        // Modifier showTab pour charger l'onglet ordre
        const originalShowTab = showTab;
        showTab = function(tabName) {
            originalShowTab(tabName);
            if (tabName === 'ordre') {
                populateOrdreCategories();
                loadOrdreList();
            }
        };
    </script>
</body>
</html>