/**
 * Renderers pour chaque vue de l'application
 */

const ViewRenderers = {
    /**
     * Vue Homepage - Tous les styles avec filtres
     */
    renderHomepage(app, state) {
        // Compter les styles par catégorie
        const categoryCount = {};
        state.allStyles.forEach(style => {
            categoryCount[style.categorie_slug] = (categoryCount[style.categorie_slug] || 0) + 1;
        });
        
        // Titre dynamique selon recherche, favoris ou normal
        let searchTitle, searchSubtitle;
        if (state.showingFavoris) {
            searchTitle = `❤️ Mes favoris`;
            searchSubtitle = `${state.styles.length} style(s) en favoris`;
        } else if (state.searchQuery) {
            searchTitle = `🔍 Résultats pour "${Utils.escapeHtml(state.searchQuery)}"`;
            searchSubtitle = `${state.styles.length} style(s) trouvé(s)`;
        } else {
            searchTitle = 'Explorez tous nos styles';
            searchSubtitle = 'Transformez vos photos en œuvres d\'art personnalisées';
        }
        
        app.innerHTML = `
            <h1 class="section-title fade-in">${searchTitle}</h1>
            <p class="section-subtitle fade-in">${searchSubtitle}</p>
            
            <!-- Filtres par catégorie -->
            <div class="category-filters fade-in">
                <button class="filter-btn ${!state.selectedCategory || state.selectedCategory === 'all' ? 'active' : ''}" 
                        onclick="NavigationManager.filterByCategory('all')">
                    🎨 Tous les styles
                    <span class="filter-count">${state.allStyles.length}</span>
                </button>
                ${state.categories.map(cat => {
                    const icon = Utils.getCategoryIcon(cat.slug);
                    const count = categoryCount[cat.slug] || 0;
                    return `
                        <button class="filter-btn ${state.selectedCategory === cat.slug ? 'active' : ''}" 
                                onclick="NavigationManager.filterByCategory('${cat.slug}')">
                            ${icon} ${cat.nom}
                            <span class="filter-count">${count}</span>
                        </button>
                    `;
                }).join('')}
            </div>
            
            <!-- Grille de styles avec effet hover -->
            <div class="styles-grid">
                ${state.styles.length === 0 ? '<div class="loading"><div class="loading-spinner"></div><div class="loading-text">Chargement des styles...</div></div>' : ''}
                ${state.styles.map(style => {
                    let imageUrl = style.url || `https://img-conseil.duia.eu/poster/styles/${style.titre.replace(/\s+/g, '_')}.png`;
                    let imageUrl2 = style.url2 || imageUrl;
                    const isSelected = state.selectedStyle && state.selectedStyle.style_id === style.style_id;
                    const isFavori = FavorisManager.isFavori(style.style_id);
                    return `
                        <div class="style-card fade-in ${isSelected ? 'selected' : ''}" onclick='NavigationManager.selectStyle(${Utils.toDataAttribute(style)})'>
                            <div class="style-image hover-effect">
                                <div class="frame-container">
                                    <img src="${imageUrl}" 
                                         data-url2="${imageUrl2}" 
                                         alt="${Utils.escapeHtml(style.titre)}"
                                         class="style-img-hover">
                                </div>
                                <button class="favori-heart ${isFavori ? 'active' : ''}" 
                                        data-favori-style="${style.style_id}"
                                        onclick="FavorisManager.toggleFavori(${style.style_id}, event)"
                                        title="${isFavori ? 'Retirer des favoris' : 'Ajouter aux favoris'}">
                                    ${isFavori ? '❤️' : '🤍'}
                                </button>
                            </div>
                            <div class="style-body">
                                <div class="style-category">${Utils.escapeHtml(style.categorie_nom)}</div>
                                <h3 class="style-title">${Utils.escapeHtml(style.titre)}</h3>
                                <p class="style-description">${Utils.escapeHtml(style.descriptif)}</p>
                            </div>
                        </div>
                    `;
                }).join('')}
            </div>
        `;
        
        // Ajouter les événements hover après le rendu
        this.setupHoverEffect();
    },
    
    /**
     * Vue Categories
     */
    renderCategories(app, state) {
        app.innerHTML = `
            <h1 class="section-title fade-in">Explorez nos catégories</h1>
            <p class="section-subtitle fade-in">Choisissez parmi nos différentes collections de posters personnalisés</p>
            <div class="categories-grid" id="categoriesGrid">
                ${state.categories.length === 0 ? '<div class="loading"><div class="loading-spinner"></div><div class="loading-text">Chargement des catégories...</div></div>' : ''}
                ${state.categories.map(cat => {
                    const icon = Utils.getCategoryIcon(cat.slug);
                    const imageContent = cat.url_img_categorie 
                        ? `<img src="${cat.url_img_categorie}" alt="${Utils.escapeHtml(cat.nom)}">` 
                        : icon;
                    return `
                        <div class="category-card fade-in" onclick='NavigationManager.selectCategory(${Utils.toDataAttribute(cat)})'>
                            <div class="category-image">${imageContent}</div>
                            <div class="category-body">
                                <h3 class="category-title">${Utils.escapeHtml(cat.nom)}</h3>
                                <p class="category-description">${Utils.escapeHtml(cat.description || 'Découvrez nos styles uniques')}</p>
                                <div class="category-footer">
                                    <span class="category-count">${cat.nb_styles} style${cat.nb_styles > 1 ? 's' : ''}</span>
                                    <div class="category-arrow">→</div>
                                </div>
                            </div>
                        </div>
                    `;
                }).join('')}
            </div>
        `;
    },
    
    /**
     * Vue Styles d'une catégorie
     */
    renderStyles(app, state) {
        app.innerHTML = `
            <button class="btn btn-back" onclick="NavigationManager.backToCategories()">← Retour aux catégories</button>
            <h1 class="section-title fade-in">${Utils.escapeHtml(state.selectedCategory.nom)}</h1>
            <p class="section-subtitle fade-in">${Utils.escapeHtml(state.selectedCategory.description || 'Choisissez le style qui vous inspire')}</p>
            <div class="styles-grid">
                ${state.styles.map(style => {
                    let imageUrl = style.url || `https://img-conseil.duia.eu/poster/styles/${style.titre.replace(/\s+/g, '_')}.png`;
                    let imageUrl2 = style.url2 || imageUrl;
                    const isSelected = state.selectedStyle && state.selectedStyle.style_id === style.style_id;
                    return `
                        <div class="style-card fade-in ${isSelected ? 'selected' : ''}" onclick='NavigationManager.selectStyle(${Utils.toDataAttribute(style)})'>
                            <div class="style-image hover-effect">
                                <div class="frame-container">
                                    <img src="${imageUrl}" 
                                         data-url2="${imageUrl2}"
                                         alt="${Utils.escapeHtml(style.titre)}"
                                         class="style-img-hover">
                                </div>
                            </div>
                            <div class="style-body">
                                <div class="style-category">${Utils.escapeHtml(style.categorie_nom)}</div>
                                <h3 class="style-title">${Utils.escapeHtml(style.titre)}</h3>
                                <p class="style-description">${Utils.escapeHtml(style.descriptif)}</p>
                            </div>
                        </div>
                    `;
                }).join('')}
            </div>
        `;
        
        // Ajouter les événements hover
        this.setupHoverEffect();
    },
    
    /**
     * Configure l'effet hover sur les images
     */
    setupHoverEffect() {
        setTimeout(() => {
            document.querySelectorAll('.style-img-hover').forEach(img => {
                const url1 = img.src;
                const url2 = img.dataset.url2;
                
                if (url2 && url2 !== url1) {
                    img.addEventListener('mouseenter', () => {
                        img.src = url2;
                    });
                    
                    img.addEventListener('mouseleave', () => {
                        img.src = url1;
                    });
                }
            });
        }, 100);
    }
};
