/**
 * Gestionnaire de sélection des versions
 */

const VersionManager = {
    selectedVersions: new Set(),
    
    /**
     * Initialise le gestionnaire
     */
    init() {
        this.selectedVersions.clear();
        this.updateUI();
    },
    
    /**
     * Toggle une version
     */
    toggleVersion(index) {
        if (this.selectedVersions.has(index)) {
            this.selectedVersions.delete(index);
        } else {
            this.selectedVersions.add(index);
        }
        this.updateUI();
    },
    
    /**
     * Sélectionner toutes les versions
     */
    selectAll() {
        const state = StateManager.getState();
        
        if (this.selectedVersions.size === state.generatedImages.length) {
            // Tout désélectionner
            this.selectedVersions.clear();
            state.generatedImages.forEach((_, index) => {
                const checkbox = document.getElementById(`select-version-${index}`);
                if (checkbox) checkbox.checked = false;
            });
        } else {
            // Tout sélectionner
            state.generatedImages.forEach((_, index) => {
                this.selectedVersions.add(index);
                const checkbox = document.getElementById(`select-version-${index}`);
                if (checkbox) checkbox.checked = true;
            });
        }
        
        this.updateUI();
    },
    
    /**
     * Met à jour l'interface
     */
    updateUI() {
        const count = this.selectedVersions.size;
        const state = StateManager.getState();
        
        // Mettre à jour le texte de sélection
        const selectionInfo = document.getElementById('selectionInfo');
        const selectionCount = selectionInfo?.querySelector('.selection-count');
        if (selectionCount) {
            if (count === 0) {
                selectionCount.textContent = 'Aucune version sélectionnée';
                selectionCount.style.color = '#94a3b8';
            } else if (count === 1) {
                selectionCount.textContent = '1 version sélectionnée';
                selectionCount.style.color = '#4ade80';
            } else {
                selectionCount.textContent = `${count} versions sélectionnées`;
                selectionCount.style.color = '#4ade80';
            }
        }
        
        // Activer/désactiver le bouton commander
        const commanderBtn = document.getElementById('commanderBtn');
        if (commanderBtn) {
            commanderBtn.disabled = count === 0;
            if (count > 0) {
                commanderBtn.innerHTML = `📋 Choisir les formats pour ${count} version${count > 1 ? 's' : ''}`;
            } else {
                commanderBtn.innerHTML = '📋 Choisir les formats d\'impression';
            }
        }
        
        // Mettre à jour le bouton "Tout sélectionner"
        const selectAllBtn = document.getElementById('selectAllBtn');
        if (selectAllBtn) {
            if (count === state.generatedImages.length) {
                selectAllBtn.innerHTML = '✗ Tout désélectionner';
            } else {
                selectAllBtn.innerHTML = '✓ Tout sélectionner';
            }
        }
        
        // Ajouter la classe selected aux versions sélectionnées
        state.generatedImages.forEach((_, index) => {
            const versionItem = document.getElementById(`version-${index}`);
            if (versionItem) {
                if (this.selectedVersions.has(index)) {
                    versionItem.classList.add('selected');
                } else {
                    versionItem.classList.remove('selected');
                }
            }
        });
    },
    
    /**
     * Aller à la page de sélection des formats
     */
    goToFormatSelection() {
        const state = StateManager.getState();
        
        if (this.selectedVersions.size === 0) {
            Utils.showToast('⚠️ Veuillez sélectionner au moins une version');
            return;
        }
        
        // Stocker les versions sélectionnées
        const selectedImages = Array.from(this.selectedVersions).map(index => ({
            index: index,
            image: state.generatedImages[index]
        }));
        
        StateManager.setState({ 
            selectedImagesForFormats: selectedImages,
            view: 'format-selection'
        });
        
        RenderManager.render();
    }
};
