/**
 * Fonctions utilitaires
 */

const Utils = {
    /**
     * Formate un prix en euros
     */
    formatPrice(price) {
        return price.toFixed(2).replace('.', ',') + ' €';
    },
    
    /**
     * Affiche un toast notification (désactivé)
     */
    showToast(message) {
        // Toast désactivé - juste log en console
        console.log('ℹ️', message);
    },
    
    /**
     * Échappe les caractères HTML
     */
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    },
    
    /**
     * Convertit un objet en attribut data JSON échappé
     */
    toDataAttribute(obj) {
        return JSON.stringify(obj).replace(/"/g, '&quot;');
    },
    
    /**
     * Récupère l'icône d'une catégorie
     */
    getCategoryIcon(slug) {
        return CONFIG.CATEGORY_ICONS[slug] || '🎨';
    },
    
    /**
     * Scroll vers le haut de la page
     */
    scrollToTop(behavior = 'smooth') {
        window.scrollTo({ top: 0, behavior });
    },
    
    /**
     * Valide une adresse email
     */
    isValidEmail(email) {
        const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    },
    
    /**
     * Valide un numéro de téléphone français
     */
    isValidPhone(phone) {
        const re = /^(?:(?:\+|00)33|0)\s*[1-9](?:[\s.-]*\d{2}){4}$/;
        return re.test(phone);
    },
    
    /**
     * Débounce une fonction
     */
    debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    },
    
    /**
     * Copie du texte dans le presse-papiers
     */
    async copyToClipboard(text) {
        try {
            await navigator.clipboard.writeText(text);
            this.showToast('✅ Copié dans le presse-papiers');
            return true;
        } catch (err) {
            console.error('Erreur copie:', err);
            return false;
        }
    },
    
    /**
     * Télécharge un fichier
     */
    downloadFile(url, filename) {
        const a = document.createElement('a');
        a.href = url;
        a.download = filename;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
    },
    
    /**
     * Formate une date
     */
    formatDate(date) {
        return new Intl.DateTimeFormat('fr-FR', {
            day: '2-digit',
            month: '2-digit',
            year: 'numeric'
        }).format(new Date(date));
    },
    
    /**
     * Génère un ID unique
     */
    generateId() {
        return Date.now().toString(36) + Math.random().toString(36).substring(2);
    }
};

// Exposer showToast globalement pour auth.js et autres modules
window.showToast = (message) => Utils.showToast(message);


// Protection contre le clic droit sur les images
document.addEventListener('DOMContentLoaded', () => {
    // Désactiver le clic droit sur toutes les images
    document.addEventListener('contextmenu', (e) => {
        if (e.target.tagName === 'IMG') {
            e.preventDefault();
            return false;
        }
    });
    
    // Désactiver le glisser-déposer des images
    document.addEventListener('dragstart', (e) => {
        if (e.target.tagName === 'IMG') {
            e.preventDefault();
            return false;
        }
    });
});