/**
 * Gestionnaire d'upload et de génération d'images
 */

const UploadManager = {
    /**
     * Gère la sélection du fichier principal
     */
    async handleFileSelectMain(event) {
        const file = event.target.files[0];
        if (!file) return;
        
        // Vérifier la taille (10 MB max)
        if (file.size > 10 * 1024 * 1024) {
            Utils.showToast('❌ L\'image ne doit pas dépasser 10 MB');
            return;
        }
        
        // Si c'est un HEIC, le convertir
        if (file.type === 'image/heic' || file.name.toLowerCase().endsWith('.heic')) {
            console.log('🔄 Conversion HEIC détectée...');
            await this.convertHeicToJpeg(file);
            return;
        }
        
        // Vérifier le type pour les autres formats
        if (!file.type.startsWith('image/')) {
            Utils.showToast('❌ Veuillez sélectionner une image');
            return;
        }
        
        // Créer un aperçu
        const reader = new FileReader();
        reader.onload = (e) => {
            StateManager.setState({
                uploadedFile: file,
                uploadedPreview: e.target.result
            });
            RenderManager.render();
            
            // Activer les boutons de génération
            [0, 1, 2].forEach(index => {
                const btn = document.getElementById(`btn-generate-${index}`);
                if (btn) btn.disabled = false;
            });
        };
        reader.readAsDataURL(file);
        
        console.log('📁 Fichier principal sélectionné:', file.name);
    },
    
    /**
     * Convertit un fichier HEIC en JPEG via le webhook
     */
    async convertHeicToJpeg(heicFile) {
        try {
            Utils.showToast('🔄 Conversion HEIC en cours...');
            
            const formData = new FormData();
            formData.append('image', heicFile);
            
            const response = await fetch(CONFIG.WEBHOOK_HEIC, {
                method: 'POST',
                body: formData
            });
            
            if (!response.ok) {
                throw new Error('Erreur lors de la conversion HEIC');
            }
            
            // Récupérer le blob JPEG converti
            const jpegBlob = await response.blob();
            
            // Créer un nouveau fichier
            const jpegFile = new File([jpegBlob], heicFile.name.replace(/\.heic$/i, '.jpg'), {
                type: 'image/jpeg'
            });
            
            console.log('✅ HEIC converti en JPEG:', jpegFile.name);
            
            // Créer un aperçu
            const reader = new FileReader();
            reader.onload = (e) => {
                StateManager.setState({
                    uploadedFile: jpegFile,
                    uploadedPreview: e.target.result
                });
                RenderManager.render();
                
                // Activer les boutons
                [0, 1, 2].forEach(index => {
                    const btn = document.getElementById(`btn-generate-${index}`);
                    if (btn) btn.disabled = false;
                });
                
                Utils.showToast('✅ Image convertie avec succès');
            };
            reader.readAsDataURL(jpegFile);
            
        } catch (error) {
            console.error('❌ Erreur conversion HEIC:', error);
            Utils.showToast('❌ Erreur lors de la conversion HEIC');
        }
    },
    
    /**
     * Génère une version depuis le formulaire principal
     */
    async generateVersionFromMain(slotIndex) {
        console.log('🚀 generateVersionFromMain appelé - slotIndex:', slotIndex);
        
        const state = StateManager.getState();
        console.log('📊 État actuel - view:', state.view);
        
        if (!state.uploadedFile) {
            Utils.showToast('❌ Veuillez d\'abord sélectionner une photo');
            return;
        }
        
        // Récupérer les valeurs actuelles du formulaire
        const titleInput = document.getElementById('image-title-main');
        const sceneInput = document.getElementById('scene-description-main');
        
        const title = titleInput ? titleInput.value : '';
        const scene = sceneInput ? sceneInput.value : '';
        const background = state.backgroundType;
        
        // Mettre à jour l'état SANS changer la vue
        StateManager.setState({
            imageTitle: title,
            sceneDescription: scene,
            backgroundType: background
        });
        
        console.log('📊 État après setState - view:', StateManager.getState().view);
        
        // NE PAS CHANGER DE VUE - Juste mettre le spinner dans le slot
        const slotContainer = document.getElementById(`version-slot-${slotIndex}`);
        console.log('🔍 Slot container trouvé:', slotContainer ? 'OUI' : 'NON');
        
        if (slotContainer) {
            console.log('✏️ Modification du slot innerHTML...');
            slotContainer.innerHTML = `
                <div class="version-header">
                    <span class="version-badge generating">⏳ Version ${slotIndex + 1} - En cours...</span>
                </div>
                <div class="generating-slot-inline">
                    <div class="spinner"></div>
                    <div class="generating-text" id="spinnerText-main">Réflexion...</div>
                </div>
            `;
            slotContainer.classList.add('generating');
            console.log('✅ Slot modifié avec spinner');
        } else {
            console.error('❌ Slot container non trouvé! ID cherché: version-slot-' + slotIndex);
        }
        
        console.log('📊 Vue actuelle après modification slot:', StateManager.getState().view);
        
        try {
            // Incrémenter le compteur
            const newGenerationCount = state.generationCount + 1;
            StateManager.setState({ generationCount: newGenerationCount });
            
            console.log('📊 Vue après incrémentation:', StateManager.getState().view);
            
            // Préparer les données
            const fd = new FormData();
            fd.append('image', state.uploadedFile);
            fd.append('style_id', state.selectedStyle.style_id);
            fd.append('style_name', state.selectedStyle.titre);
            fd.append('prompt', state.selectedStyle.prompt);
            fd.append('background_type', background);
            fd.append('session_id', state.sessionId);
            fd.append('generation_count', newGenerationCount);
            
            if (title) fd.append('title', title);
            if (scene) fd.append('scene_description', scene);
            
            console.log('📤 Génération version', slotIndex + 1);
            console.log('  - session_id:', state.sessionId);
            console.log('  - generation_count:', newGenerationCount);
            console.log('  - background_type:', background);
            console.log('  - title:', title);
            console.log('  - scene:', scene);
            
            // Envoyer au webhook
            fetch(CONFIG.WEBHOOK_GENERATE, {
                method: 'POST',
                body: fd
            }).catch(err => console.log('Webhook error:', err));
            
            // Construire l'URL
            const imageUrl = `${CONFIG.IMAGE_BASE_URL}/${state.sessionId}/ImgRendu_${state.sessionId}_v${newGenerationCount}.jpeg`;
            console.log('🔍 URL attendue:', imageUrl);
            
            // Polling avec mise à jour du texte dans le slot
            const success = await this.pollForImageInSlotMain(imageUrl, state.sessionId, newGenerationCount);
            
            if (success) {
                // Re-render pour afficher le résultat
                RenderManager.render({ noScroll: true });
                Utils.showToast(`✅ Version ${slotIndex + 1} générée !`);
            } else {
                throw new Error('Timeout - Image non trouvée');
            }
            
        } catch (error) {
            console.error('❌ Erreur génération:', error);
            Utils.showToast('❌ ' + error.message);
            
            // Re-render pour remettre le formulaire
            RenderManager.render({ noScroll: true });
        }
    },
    
    /**
     * Polling pour la génération depuis le formulaire principal
     */
    async pollForImageInSlotMain(imageUrl, sessionId, version) {
        const maxTries = 60;
        
        for (let i = 0; i < maxTries; i++) {
            let statusText = '';
            
            if (i < 3) {
                statusText = 'Contrôle des données...';
            } else if (i < 6) {
                statusText = 'Réflexion...';
            } else {
                const progress = i - 5;
                statusText = `Génération... ${progress}/54`;
            }
            
            // Mettre à jour le texte dans le slot
            const spinnerText = document.getElementById('spinnerText-main');
            if (spinnerText) {
                spinnerText.textContent = statusText;
            }
            
            console.log(`🔍 Tentative ${i + 1}/${maxTries} - ${statusText}`);
            
            // Tester si l'image existe
            const exists = await this.imageExists(imageUrl);
            
            if (exists) {
                console.log('✅ Image trouvée !');
                
                const imageUrlWithCache = imageUrl + '?t=' + Date.now();
                
                // Ajouter la version au tableau
                const state = StateManager.getState();
                const newImages = [...state.generatedImages, {
                    url: imageUrlWithCache,
                    version: version,
                    sessionId: sessionId
                }];
                
                StateManager.setState({
                    generatedImages: newImages,
                    currentVersion: newImages.length - 1
                });
                
                return true;
            }
            
            // Attendre 3 secondes
            await new Promise(r => setTimeout(r, 3000));
        }
        
        return false;
    },
    
    /**
     * Gère la sélection d'un fichier pour un slot spécifique
     */
    handleFileSelectSlot(event, slotIndex) {
        const file = event.target.files[0];
        if (!file) return;
        
        // Vérifier le type
        if (!file.type.startsWith('image/')) {
            Utils.showToast('❌ Veuillez sélectionner une image');
            return;
        }
        
        // Vérifier la taille (10 MB max)
        if (file.size > 10 * 1024 * 1024) {
            Utils.showToast('❌ L\'image ne doit pas dépasser 10 MB');
            return;
        }
        
        // Créer un aperçu
        const reader = new FileReader();
        reader.onload = (e) => {
            const previewDiv = document.getElementById(`upload-preview-${slotIndex}`);
            if (previewDiv) {
                previewDiv.innerHTML = `<img src="${e.target.result}" alt="Aperçu">`;
            }
            
            // Activer le bouton générer
            const btnGenerate = document.getElementById(`btn-generate-${slotIndex}`);
            if (btnGenerate) {
                btnGenerate.disabled = false;
            }
        };
        reader.readAsDataURL(file);
        
        // Stocker le fichier dans un objet temporaire
        if (!window.slotFiles) window.slotFiles = {};
        window.slotFiles[slotIndex] = {
            file: file,
            title: '',
            scene: '',
            background: 'ia'
        };
        
        console.log('📁 Fichier sélectionné pour slot', slotIndex, ':', file.name);
    },
    
    /**
     * Génère l'image pour un slot spécifique
     */
    async generateImageSlot(slotIndex) {
        const state = StateManager.getState();
        
        if (!window.slotFiles || !window.slotFiles[slotIndex]) {
            Utils.showToast('❌ Veuillez d\'abord sélectionner une photo');
            return;
        }
        
        const slotData = window.slotFiles[slotIndex];
        
        // Récupérer les valeurs des champs
        const titleInput = document.getElementById(`title-${slotIndex}`);
        const sceneInput = document.getElementById(`scene-${slotIndex}`);
        const bgButtons = document.querySelectorAll(`#generate-slot-${slotIndex} .bg-option-sm`);
        let backgroundType = 'ia';
        bgButtons.forEach(btn => {
            if (btn.classList.contains('active')) {
                backgroundType = btn.getAttribute('data-bg');
            }
        });
        
        slotData.title = titleInput ? titleInput.value : '';
        slotData.scene = sceneInput ? sceneInput.value : '';
        slotData.background = backgroundType;
        
        // NE PAS CHANGER DE VUE - juste modifier le slot visuellement
        const slotContainer = document.getElementById(`generate-slot-${slotIndex}`);
        if (slotContainer) {
            // Remplacer le contenu du slot par un spinner
            slotContainer.innerHTML = `
                <div class="generating-slot-inline">
                    <div class="spinner"></div>
                    <div class="generating-text" id="spinnerText-${slotIndex}">Réflexion...</div>
                </div>
            `;
            slotContainer.classList.add('generating');
        }
        
        try {
            // Incrémenter le compteur
            const newGenerationCount = state.generationCount + 1;
            StateManager.setState({ generationCount: newGenerationCount });
            
            // Préparer les données
            const fd = new FormData();
            fd.append('image', slotData.file);
            fd.append('style_id', state.selectedStyle.style_id);
            fd.append('style_name', state.selectedStyle.titre);
            fd.append('prompt', state.selectedStyle.prompt);
            fd.append('background_type', slotData.background);
            fd.append('session_id', state.sessionId);
            fd.append('generation_count', newGenerationCount);
            
            if (slotData.title) {
                fd.append('title', slotData.title);
            }
            
            if (slotData.scene) {
                fd.append('scene_description', slotData.scene);
            }
            
            console.log('📤 Génération slot', slotIndex);
            console.log('  - session_id:', state.sessionId);
            console.log('  - generation_count:', newGenerationCount);
            console.log('  - background_type:', slotData.background);
            
            // Envoyer au webhook
            fetch(CONFIG.WEBHOOK_GENERATE, {
                method: 'POST',
                body: fd
            }).catch(err => console.log('Webhook error:', err));
            
            // Construire l'URL de l'image attendue
            const imageUrl = `${CONFIG.IMAGE_BASE_URL}/${state.sessionId}/ImgRendu_${state.sessionId}_v${newGenerationCount}.jpeg`;
            console.log('🔍 URL attendue:', imageUrl);
            
            // Polling avec mise à jour du texte dans le slot
            const success = await this.pollForImageInSlot(imageUrl, state.sessionId, newGenerationCount, slotIndex);
            
            if (success) {
                // Nettoyer le slot
                delete window.slotFiles[slotIndex];
                
                // Mettre à jour JUSTE le slot avec l'image générée (pas de render complet)
                this.updateSlotWithImage(slotIndex, imageUrl + '?t=' + Date.now(), newGenerationCount);
                
                Utils.showToast('✅ Image générée !');
            } else {
                throw new Error('Timeout - Image non trouvée');
            }
            
        } catch (error) {
            console.error('❌ Erreur génération:', error);
            Utils.showToast('❌ ' + error.message);
            
            // Remettre le formulaire dans le slot
            this.resetSlotToForm(slotIndex);
        }
    },
    
    /**
     * Met à jour un slot avec l'image générée (sans re-render complet)
     */
    updateSlotWithImage(slotIndex, imageUrl, version) {
        const slotContainer = document.getElementById(`generate-slot-${slotIndex}`);
        if (!slotContainer) return;
        
        slotContainer.classList.remove('generating');
        slotContainer.innerHTML = `
            <div class="generated-image-slot">
                <div class="version-image-container">
                    <div class="watermarked-image">
                        <img src="${imageUrl}" alt="Version ${version}" class="version-img">
                        <div class="watermark-overlay"></div>
                    </div>
                </div>
                <div class="version-checkbox-container">
                    <label class="checkbox-wrapper" onclick="VersionManager.toggleVersion(${StateManager.getState().generatedImages.length - 1})">
                        <input type="checkbox" class="version-checkbox" id="check-${StateManager.getState().generatedImages.length - 1}">
                        <span class="checkbox-custom"></span>
                        <span class="checkbox-label-text">Sélectionner cette version</span>
                    </label>
                </div>
            </div>
        `;
        
        // Mettre à jour le VersionManager si nécessaire
        if (typeof VersionManager !== 'undefined') {
            VersionManager.updateUI();
        }
    },
    
    /**
     * Remet le formulaire dans un slot après erreur
     */
    resetSlotToForm(slotIndex) {
        const slotContainer = document.getElementById(`generate-slot-${slotIndex}`);
        if (!slotContainer) return;
        
        const state = StateManager.getState();
        slotContainer.classList.remove('generating');
        slotContainer.innerHTML = `
            <div class="slot-upload-zone" onclick="document.getElementById('slot-file-${slotIndex}').click()">
                <input type="file" id="slot-file-${slotIndex}" accept="image/*" 
                       onchange="UploadManager.handleSlotFile(event, ${slotIndex})" hidden>
                <div class="slot-upload-content">
                    <span class="slot-upload-icon">📷</span>
                    <span class="slot-upload-text">Cliquez pour ajouter une photo</span>
                </div>
            </div>
        `;
    },
    
    /**
     * Polling pour vérifier l'image - version pour slot inline
     */
    async pollForImageInSlot(imageUrl, sessionId, version, slotIndex) {
        const maxTries = 60;
        
        for (let i = 0; i < maxTries; i++) {
            let statusText = '';
            
            if (i < 3) {
                statusText = 'Contrôle des données...';
            } else if (i < 6) {
                statusText = 'Réflexion...';
            } else {
                const progress = i - 5;
                statusText = `Génération... ${progress}/54`;
            }
            
            // Mettre à jour le texte dans le slot
            const spinnerText = document.getElementById(`spinnerText-${slotIndex}`);
            if (spinnerText) {
                spinnerText.textContent = statusText;
            }
            
            console.log(`🔍 Slot ${slotIndex} - Tentative ${i + 1}/${maxTries} - ${statusText}`);
            
            // Tester si l'image existe
            const exists = await this.imageExists(imageUrl);
            
            if (exists) {
                console.log('✅ Image trouvée !');
                
                const imageUrlWithCache = imageUrl + '?t=' + Date.now();
                
                // Ajouter la version au tableau
                const state = StateManager.getState();
                const newImages = [...state.generatedImages, {
                    url: imageUrlWithCache,
                    version: version,
                    sessionId: sessionId
                }];
                
                StateManager.setState({
                    generatedImages: newImages,
                    currentVersion: newImages.length - 1
                });
                
                return true;
            }
            
            // Attendre 3 secondes
            await new Promise(r => setTimeout(r, 3000));
        }
        
        return false;
    },
    
    /**
     * Traite un fichier uploadé
     */
    processFile(file) {
        // Validation
        if (!file.type.startsWith('image/')) {
            Utils.showToast('❌ Veuillez sélectionner une image');
            return;
        }
        
        if (file.size > 10 * 1024 * 1024) {
            Utils.showToast('❌ Image trop volumineuse (max 10 Mo)');
            return;
        }
        
        // Lecture du fichier
        const reader = new FileReader();
        
        reader.onload = (e) => {
            StateManager.setState({
                uploadedFile: file,
                uploadedPreview: e.target.result
            });
            RenderManager.render();
        };
        
        reader.onerror = () => {
            Utils.showToast('❌ Erreur lors de la lecture du fichier');
        };
        
        reader.readAsDataURL(file);
    },
    
    /**
     * Supprime le fichier uploadé
     */
    removeFile() {
        StateManager.setState({
            uploadedFile: null,
            uploadedPreview: null
        });
        RenderManager.render();
    },
    
    /**
     * Configure le drag & drop
     */
    setupDragAndDrop() {
        const uploadZone = document.getElementById('uploadZone');
        if (!uploadZone) return;
        
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            uploadZone.addEventListener(eventName, (e) => {
                e.preventDefault();
                e.stopPropagation();
            });
        });
        
        ['dragenter', 'dragover'].forEach(eventName => {
            uploadZone.addEventListener(eventName, () => {
                uploadZone.classList.add('dragover');
            });
        });
        
        ['dragleave', 'drop'].forEach(eventName => {
            uploadZone.addEventListener(eventName, () => {
                uploadZone.classList.remove('dragover');
            });
        });
        
        uploadZone.addEventListener('drop', (e) => {
            const files = e.dataTransfer.files;
            if (files.length > 0) {
                this.processFile(files[0]);
            }
        });
    },
    
    /**
     * Génère l'image avec l'IA
     */
    async generateImage() {
        const state = StateManager.getState();
        
        if (!state.uploadedFile) {
            Utils.showToast('❌ Veuillez d\'abord sélectionner une image');
            return;
        }
        
        // Passer à la vue de génération
        StateManager.setState({ view: 'generating' });
        RenderManager.render({ noScroll: true });
        
        try {
            // Incrémenter le compteur
            const newGenerationCount = state.generationCount + 1;
            StateManager.setState({ generationCount: newGenerationCount });
            
            // Préparer les données
            const fd = new FormData();
            fd.append('image', state.uploadedFile);
            fd.append('style_id', state.selectedStyle.style_id);
            fd.append('style_name', state.selectedStyle.titre);
            fd.append('prompt', state.selectedStyle.prompt);
            fd.append('background_type', state.backgroundType);
            fd.append('session_id', state.sessionId);
            fd.append('generation_count', newGenerationCount);
            
            if (state.imageTitle) {
                fd.append('title', state.imageTitle);
            }
            
            if (state.sceneDescription) {
                fd.append('scene_description', state.sceneDescription);
            }
            
            // Log
            console.log('📤 Envoi au webhook:', CONFIG.WEBHOOK_GENERATE);
            console.log('  - session_id:', state.sessionId);
            console.log('  - generation_count:', newGenerationCount);
            console.log('  - background_type:', state.backgroundType);
            
            // Envoyer au webhook (fire and forget)
            fetch(CONFIG.WEBHOOK_GENERATE, {
                method: 'POST',
                body: fd
            }).catch(err => console.log('Webhook error:', err));
            
            // Construire l'URL de l'image attendue
            const imageUrl = `${CONFIG.IMAGE_BASE_URL}/${state.sessionId}/ImgRendu_${state.sessionId}_v${newGenerationCount}.jpeg`;
            console.log('🔍 URL attendue:', imageUrl);
            
            // Polling pour vérifier l'image
            const success = await this.pollForImage(imageUrl, state.sessionId, newGenerationCount);
            
            if (!success) {
                throw new Error('Timeout - Image non trouvée après 3 minutes');
            }
            
        } catch (error) {
            console.error('❌ Erreur génération:', error);
            Utils.showToast('❌ ' + error.message);
            
            // Retour à l'upload
            StateManager.setState({ view: 'upload' });
            RenderManager.render();
        }
    },
    
    /**
     * Polling pour vérifier l'existence de l'image
     */
    async pollForImage(imageUrl, sessionId, version) {
        const maxTries = 60; // 60 tentatives
        
        for (let i = 0; i < maxTries; i++) {
            // Déterminer le texte selon l'étape
            let statusText = '';
            let shortText = '';
            
            if (i < 3) {
                statusText = '🔍 Contrôle des données...';
                shortText = 'Contrôle des données...';
            } else if (i < 6) {
                statusText = '🧠 Réflexion en cours...';
                shortText = 'Réflexion...';
            } else {
                const progress = i - 5;
                statusText = `🎨 Génération de l'image... (${progress}/54)`;
                shortText = `Génération... ${progress}/54`;
            }
            
            console.log(`🔍 Tentative ${i + 1}/${maxTries} - ${statusText}`);
            
            // Mettre à jour TOUS les éléments de texte possibles
            const elementsToUpdate = [
                { id: 'generationText', text: statusText },
                { id: 'spinnerText', text: shortText }
            ];
            
            elementsToUpdate.forEach(({ id, text }) => {
                const el = document.getElementById(id);
                if (el) {
                    el.textContent = text;
                    console.log(`✅ Mis à jour #${id}: ${text}`);
                }
            });
            
            // Aussi mettre à jour par classe
            document.querySelectorAll('.generating-text').forEach(el => {
                el.textContent = shortText;
            });
            
            // Tester si l'image existe
            const exists = await this.imageExists(imageUrl);
            
            if (exists) {
                console.log('✅ Image trouvée !');
                
                const imageUrlWithCache = imageUrl + '?t=' + Date.now();
                
                // Ajouter la version au tableau
                const state = StateManager.getState();
                const newImages = [...state.generatedImages, {
                    url: imageUrlWithCache,
                    version: version,
                    sessionId: sessionId
                }];
                
                // Succès - toujours aller sur result pour afficher les 3 zones
                StateManager.setState({
                    generatedImages: newImages,
                    currentVersion: newImages.length - 1,
                    view: 'result'  // Toujours afficher la vue result
                });
                
                RenderManager.render({ noScroll: true });
                return true;
            }
            
            // Attendre 3 secondes avant de réessayer
            await new Promise(r => setTimeout(r, 3000));
        }
        
        // Timeout
        console.error('❌ Timeout après', maxTries, 'tentatives');
        return false;
    },
    
    /**
     * Teste si une image existe
     */
    async imageExists(url) {
        return new Promise((resolve) => {
            const img = new Image();
            
            const timeout = setTimeout(() => {
                img.onload = img.onerror = null;
                resolve(false);
            }, 5000);
            
            img.onload = () => {
                clearTimeout(timeout);
                resolve(true);
            };
            
            img.onerror = () => {
                clearTimeout(timeout);
                resolve(false);
            };
            
            img.src = url + '?t=' + Date.now(); // Cache busting
        });
    }
};

// Exposer les fonctions globalement pour les événements onclick
window.handleFileSelectMain = (event) => UploadManager.handleFileSelectMain(event);
window.generateVersionFromMain = (index) => UploadManager.generateVersionFromMain(index);