/**
 * Fonctions d'upload simples - Globales et directes
 */

// Fonction principale d'upload
function handleFileSelectMain(event) {
    const file = event.target.files[0];
    if (!file) return;
    
    console.log('📁 Fichier sélectionné:', file.name, file.type);
    
    // Si HEIC, convertir
    if (file.type === 'image/heic' || file.name.toLowerCase().endsWith('.heic')) {
        convertHeicAndUpload(file);
        return;
    }
    
    // Vérifier le type
    if (!file.type.startsWith('image/')) {
        alert('❌ Veuillez sélectionner une image');
        return;
    }
    
    // Vérifier la taille
    if (file.size > 10 * 1024 * 1024) {
        alert('❌ L\'image ne doit pas dépasser 10 MB');
        return;
    }
    
    // Lire et afficher l'aperçu
    const reader = new FileReader();
    reader.onload = (e) => {
        // Mettre à jour l'état
        StateManager.setState({
            uploadedFile: file,
            uploadedPreview: e.target.result
        });
        
        // Re-render
        RenderManager.render();
        
        console.log('✅ Fichier chargé et aperçu créé');
    };
    reader.readAsDataURL(file);
}

// Conversion HEIC
async function convertHeicAndUpload(heicFile) {
    try {
        console.log('🔄 Conversion HEIC en cours...');
        
        const formData = new FormData();
        formData.append('image', heicFile);
        
        const response = await fetch(CONFIG.WEBHOOK_HEIC, {
            method: 'POST',
            body: formData
        });
        
        if (!response.ok) {
            throw new Error('Erreur lors de la conversion HEIC');
        }
        
        const jpegBlob = await response.blob();
        const jpegFile = new File([jpegBlob], heicFile.name.replace(/\.heic$/i, '.jpg'), {
            type: 'image/jpeg'
        });
        
        console.log('✅ HEIC converti:', jpegFile.name, 'Taille:', jpegBlob.size);
        
        // Lire et afficher
        const reader = new FileReader();
        reader.onload = (e) => {
            const previewDataUrl = e.target.result;
            console.log('📷 Preview chargé, longueur data URL:', previewDataUrl.length);
            
            // Mettre à jour le state
            StateManager.setState({
                uploadedFile: jpegFile,
                uploadedPreview: previewDataUrl
            });
            
            // Mettre à jour directement l'image dans le DOM
            setTimeout(() => {
                // Chercher le conteneur principal
                const previewMain = document.getElementById('upload-preview-main');
                console.log('🔍 Element upload-preview-main trouvé:', !!previewMain);
                
                if (previewMain) {
                    previewMain.innerHTML = `
                        <img src="${previewDataUrl}" alt="Aperçu" class="preview-img-main" style="max-width: 100%; max-height: 200px; object-fit: contain; display: block; margin: 0 auto;">
                    `;
                    previewMain.style.display = 'flex';
                    previewMain.style.flexDirection = 'column';
                    previewMain.style.alignItems = 'center';
                    previewMain.style.justifyContent = 'center';
                    console.log('✅ Preview main mis à jour avec image');
                }
                
                // Aussi chercher par classe
                const allPreviews = document.querySelectorAll('.upload-preview-main');
                console.log('🔍 Nombre de .upload-preview-main:', allPreviews.length);
                allPreviews.forEach((el, i) => {
                    el.innerHTML = `
                        <img src="${previewDataUrl}" alt="Aperçu" class="preview-img-main" style="max-width: 100%; max-height: 200px; object-fit: contain; display: block; margin: 0 auto;">
                    `;
                    el.style.display = 'flex';
                    el.style.flexDirection = 'column';
                    el.style.alignItems = 'center';
                    el.style.justifyContent = 'center';
                    console.log('✅ Preview', i, 'mis à jour');
                });
                
                // Activer les boutons de génération
                [0, 1, 2].forEach(index => {
                    const btn = document.getElementById(`btn-generate-${index}`);
                    if (btn) {
                        btn.disabled = false;
                        console.log(`✅ Bouton btn-generate-${index} activé`);
                    }
                });
                
                // Aussi activer le bouton principal si présent
                const mainGenerateBtn = document.querySelector('.btn-generate-main');
                if (mainGenerateBtn) {
                    mainGenerateBtn.disabled = false;
                    console.log('✅ Bouton principal de génération activé');
                }
                
            }, 200);
        };
        reader.readAsDataURL(jpegFile);
        
    } catch (error) {
        console.error('❌ Erreur conversion HEIC:', error);
        alert('❌ Erreur lors de la conversion HEIC');
    }
}

// Fonction pour générer une version - délègue à UploadManager
async function generateVersionFromMain(index) {
    console.log('🎨 generateVersionFromMain appelé depuis upload-helpers - index:', index);
    
    const state = StateManager.getState();
    
    if (!state.uploadedFile) {
        alert('❌ Veuillez d\'abord sélectionner une photo');
        return;
    }
    
    // Récupérer les valeurs du formulaire
    const titleInput = document.getElementById('image-title-main');
    const sceneInput = document.getElementById('scene-description-main');
    
    const title = titleInput ? titleInput.value : '';
    const scene = sceneInput ? sceneInput.value : '';
    const background = state.backgroundType || 'ia';
    
    // Mettre à jour l'état SANS changer la vue
    StateManager.setState({
        imageTitle: title,
        sceneDescription: scene,
        backgroundType: background
    });
    
    console.log('📊 Vue actuelle:', state.view, '- NE PAS CHANGER');
    
    // NE PAS CHANGER DE VUE - Juste mettre le spinner dans le slot
    const slotContainer = document.getElementById(`version-slot-${index}`);
    console.log('🔍 Slot container trouvé:', slotContainer ? 'OUI' : 'NON', '- ID: version-slot-' + index);
    
    if (slotContainer) {
        slotContainer.innerHTML = `
            <div class="version-header">
                <span class="version-badge generating">⏳ Version ${index + 1} - En cours...</span>
            </div>
            <div class="generating-slot-inline">
                <div class="spinner"></div>
                <div class="generating-text" id="spinnerText-main">Réflexion...</div>
            </div>
        `;
        slotContainer.classList.add('generating');
        console.log('✅ Spinner inséré dans le slot');
    }
    
    try {
        // Incrémenter le compteur
        const newGenerationCount = state.generationCount + 1;
        StateManager.setState({ generationCount: newGenerationCount });
        
        // Préparer les données
        const fd = new FormData();
        fd.append('image', state.uploadedFile);
        fd.append('style_id', state.selectedStyle.style_id);
        fd.append('style_name', state.selectedStyle.titre);
        fd.append('prompt', state.selectedStyle.prompt);
        fd.append('background_type', background);
        fd.append('session_id', state.sessionId);
        fd.append('generation_count', newGenerationCount);
        
        if (title) fd.append('title', title);
        if (scene) fd.append('scene_description', scene);
        
        console.log('📤 Envoi au webhook:', CONFIG.WEBHOOK_GENERATE);
        
        // Envoyer au webhook
        fetch(CONFIG.WEBHOOK_GENERATE, {
            method: 'POST',
            body: fd
        }).catch(err => console.log('Webhook error:', err));
        
        // Construire l'URL attendue
        const imageUrl = `${CONFIG.IMAGE_BASE_URL}/${state.sessionId}/ImgRendu_${state.sessionId}_v${newGenerationCount}.jpeg`;
        console.log('🔍 URL attendue:', imageUrl);
        
        // Polling avec mise à jour du spinner
        const success = await pollForGeneratedImageInline(imageUrl, state.sessionId, newGenerationCount);
        
        if (success) {
            // Re-render SANS scroll
            console.log('✅ Image trouvée, re-render sans scroll');
            RenderManager.render({ noScroll: true });
            
            if (typeof Utils !== 'undefined' && Utils.showToast) {
                Utils.showToast(`✅ Version ${index + 1} générée !`);
            }
        } else {
            throw new Error('Timeout - Image non trouvée');
        }
        
    } catch (error) {
        console.error('❌ Erreur génération:', error);
        alert('❌ ' + error.message);
        
        // Re-render sans scroll
        RenderManager.render({ noScroll: true });
    }
}

// Polling inline - met à jour le spinner sans changer de vue
async function pollForGeneratedImageInline(imageUrl, sessionId, version) {
    const maxTries = 60;
    
    for (let i = 0; i < maxTries; i++) {
        let statusText = '';
        
        if (i < 3) {
            statusText = 'Contrôle des données...';
        } else if (i < 6) {
            statusText = 'Réflexion...';
        } else {
            const progress = i - 5;
            statusText = `Génération... ${progress}/54`;
        }
        
        // Mettre à jour le texte du spinner
        const spinnerText = document.getElementById('spinnerText-main');
        if (spinnerText) {
            spinnerText.textContent = statusText;
        }
        
        console.log(`🔍 Tentative ${i + 1}/${maxTries} - ${statusText}`);
        console.log(`🔍 Test URL: ${imageUrl}`);
        
        // Tester si l'image existe avec la méthode Image()
        const exists = await testImageExists(imageUrl);
        
        if (exists) {
            console.log('✅ Image trouvée !');
            
            const imageUrlWithCache = imageUrl + '?t=' + Date.now();
            
            // Ajouter la version au tableau
            const state = StateManager.getState();
            const newImages = [...state.generatedImages, {
                url: imageUrlWithCache,
                version: version,
                sessionId: sessionId
            }];
            
            StateManager.setState({
                generatedImages: newImages,
                currentVersion: newImages.length - 1
            });
            
            return true;
        }
        
        // Attendre 3 secondes
        await new Promise(r => setTimeout(r, 3000));
    }
    
    return false;
}

// Fonction de polling pour l'image générée
async function pollForGeneratedImage(imageUrl, sessionId, version) {
    const maxTries = 60;
    
    for (let i = 0; i < maxTries; i++) {
        // Déterminer le texte selon l'étape
        let statusText = '';
        let shortText = '';
        
        if (i < 3) {
            statusText = '🔍 Contrôle des données...';
            shortText = 'Contrôle des données...';
        } else if (i < 6) {
            statusText = '🧠 Réflexion en cours...';
            shortText = 'Réflexion...';
        } else {
            statusText = '🎨 Génération en cours...';
            shortText = 'Génération...';
        }
        
        console.log(`🔍 Vérification ${i + 1}/${maxTries} - ${shortText}`);
        
        // Mettre à jour le texte principal
        const generationText = document.getElementById('generationText');
        if (generationText) {
            generationText.textContent = statusText;
        }
        
        // Mettre à jour tous les éléments avec la classe generating-text
        document.querySelectorAll('.generating-text').forEach(el => {
            el.textContent = shortText;
        });
        
        // Tester si l'image existe
        const exists = await testImageExists(imageUrl);
        
        if (exists) {
            console.log('✅ Image trouvée !');
            
            // Ajouter aux images générées
            const state = StateManager.getState();
            const imageUrlWithCache = imageUrl + '?t=' + Date.now();
            
            const newImages = [...state.generatedImages, {
                url: imageUrlWithCache,
                version: version,
                sessionId: sessionId
            }];
            
            StateManager.setState({
                generatedImages: newImages,
                currentVersion: newImages.length - 1
            });
            
            return true;
        }
        
        // Attendre 3 secondes
        await new Promise(r => setTimeout(r, 3000));
    }
    
    console.error('❌ Timeout après', maxTries, 'tentatives');
    return false;
}

// Teste si une image existe
async function testImageExists(url) {
    return new Promise((resolve) => {
        const img = new Image();
        
        const timeout = setTimeout(() => {
            img.onload = img.onerror = null;
            resolve(false);
        }, 5000);
        
        img.onload = () => {
            clearTimeout(timeout);
            resolve(true);
        };
        
        img.onerror = () => {
            clearTimeout(timeout);
            resolve(false);
        };
        
        img.src = url + '?t=' + Date.now();
    });
}

console.log('✅ Fonctions upload globales chargées');