/**
 * Gestionnaire des calculs de prix
 * 
 * Règle de dégressivité (par format, même image) :
 * - 1ère affiche = prix plein (prix_1er)
 * - 2ème affiche = -25% (prix_2eme)
 * - 3ème et suivantes = -40% (prix_3eme_plus)
 */

const PricingManager = {
    /**
     * Calcule le prix pour une quantité donnée d'un même format
     */
    calculate(quantite, format = 'A3') {
        const formatConfig = CONFIG.FORMATS[format];
        
        // Prix par palier (depuis la BDD ou config)
        const prix1er = formatConfig.prix_1er || formatConfig.prix;
        const prix2eme = formatConfig.prix_2eme || (formatConfig.prix * 0.75);
        const prix3emePlus = formatConfig.prix_3eme_plus || (formatConfig.prix * 0.60);
        
        let total = 0;
        let detail = {
            premiere: prix1er,
            deuxieme: prix2eme,
            troisiemePlus: prix3emePlus,
            qte_1er: 0,
            qte_2eme: 0,
            qte_3eme_plus: 0,
            total_1er: 0,
            total_2eme: 0,
            total_3eme_plus: 0
        };
        
        if (quantite >= 1) {
            // 1ère affiche
            detail.qte_1er = 1;
            detail.total_1er = prix1er;
            total += prix1er;
        }
        
        if (quantite >= 2) {
            // 2ème affiche (-25%)
            detail.qte_2eme = 1;
            detail.total_2eme = prix2eme;
            total += prix2eme;
        }
        
        if (quantite >= 3) {
            // 3ème et suivantes (-40%)
            detail.qte_3eme_plus = quantite - 2;
            detail.total_3eme_plus = detail.qte_3eme_plus * prix3emePlus;
            total += detail.total_3eme_plus;
        }
        
        const unitaire = total / quantite;
        
        return {
            total: total,
            unitaire: unitaire,
            detail: detail
        };
    },
    
    /**
     * Calcule le prix simple (sans dégressivité) - pour des images différentes
     */
    calculateSimple(quantite, format = 'A3') {
        const formatConfig = CONFIG.FORMATS[format];
        const prix = formatConfig.prix_1er || formatConfig.prix;
        const total = prix * quantite;
        
        return {
            total: total,
            unitaire: prix,
            detail: {
                prix_unitaire: prix,
                quantite: quantite
            }
        };
    },
    
    /**
     * Met à jour l'affichage des prix
     */
    updateDisplay() {
        const state = StateManager.getState();
        const pricing = this.calculate(state.quantity, state.format);
        
        const priceSummary = document.querySelector('.price-summary');
        if (priceSummary) {
            let detailHTML = '';
            
            // 1ère affiche
            detailHTML += `
                <div class="price-line">
                    <span>1ère affiche</span>
                    <strong>${Utils.formatPrice(pricing.detail.premiere)}</strong>
                </div>
            `;
            
            // 2ème affiche si qty >= 2
            if (state.quantity >= 2) {
                detailHTML += `
                    <div class="price-line">
                        <span>2ème affiche <small>(-25%)</small></span>
                        <strong>${Utils.formatPrice(pricing.detail.deuxieme)}</strong>
                    </div>
                `;
            }
            
            // 3ème et + si qty >= 3
            if (state.quantity >= 3) {
                detailHTML += `
                    <div class="price-line">
                        <span>${pricing.detail.qte_3eme_plus} suivante(s) <small>(-40%)</small></span>
                        <strong>${Utils.formatPrice(pricing.detail.total_3eme_plus)}</strong>
                    </div>
                `;
            }
            
            priceSummary.innerHTML = `
                <div class="price-line">
                    <span>Quantité</span>
                    <strong>${state.quantity} ${state.format}</strong>
                </div>
                ${detailHTML}
                <div class="price-divider"></div>
                <div class="price-line price-total">
                    <span>TOTAL TTC</span>
                    <strong>${Utils.formatPrice(pricing.total)}</strong>
                </div>
                <div class="price-unit">
                    Prix unitaire moyen : <strong>${Utils.formatPrice(pricing.unitaire)}</strong>
                </div>
            `;
        }
        
        // Mise à jour de la grille de quantités
        const rows = document.querySelectorAll('.price-table tbody tr');
        rows.forEach(row => {
            row.classList.remove('selected');
            const qtyCell = row.querySelector('.qty-col');
            if (qtyCell && parseInt(qtyCell.textContent) === state.quantity) {
                row.classList.add('selected');
            }
        });
    },
    
    /**
     * Génère le tableau des prix avec dégressivité
     */
    generatePriceTable(format = 'A3') {
        const formatConfig = CONFIG.FORMATS[format];
        const prix1er = formatConfig.prix_1er || formatConfig.prix;
        const prix2eme = formatConfig.prix_2eme || (formatConfig.prix * 0.75);
        const prix3emePlus = formatConfig.prix_3eme_plus || (formatConfig.prix * 0.60);
        
        const quantities = [1, 2, 3, 5, 10];
        
        const rows = quantities.map(qty => {
            const pricing = this.calculate(qty, format);
            return `
                <tr onclick="OrderManager.selectQuantityFromGrid(${qty})">
                    <td class="qty-col">${qty}</td>
                    <td>${Utils.formatPrice(pricing.total)}</td>
                    <td class="unit-col">${Utils.formatPrice(pricing.unitaire)}</td>
                </tr>
            `;
        }).join('');
        
        return `
            <div class="price-info-box">
                <p><strong>Tarif dégressif :</strong></p>
                <ul>
                    <li>1ère affiche : <strong>${Utils.formatPrice(prix1er)}</strong></li>
                    <li>2ème : <strong>${Utils.formatPrice(prix2eme)}</strong> <small>(-25%)</small></li>
                    <li>3ème et + : <strong>${Utils.formatPrice(prix3emePlus)}</strong> <small>(-40%)</small></li>
                </ul>
            </div>
            <table class="price-table">
                <thead>
                    <tr>
                        <th>Qté</th>
                        <th>Total TTC</th>
                        <th>Prix/u moyen</th>
                    </tr>
                </thead>
                <tbody>
                    ${rows}
                </tbody>
            </table>
        `;
    }
};
