/**
 * Gestionnaire des commandes
 */

const OrderManager = {
    formatQuantities: {}, // {format: quantity}
    currentImage: null,
    
    /**
     * Mettre à jour tous les compteurs de panier dans la page
     */
    updateAllCartCounters() {
        const count = CartManager.getCartCount();
        console.log('🔄 Mise à jour compteurs panier:', count);
        
        // Tous les IDs possibles de compteurs
        const counterIds = ['cart-count-min', 'cart-count-order', 'cart-count-display'];
        
        counterIds.forEach(id => {
            const element = document.getElementById(id);
            if (element) {
                console.log(`✅ Compteur trouvé: ${id}, valeur: ${count}`);
                element.textContent = count;
                // Force le rendu visuel
                element.style.display = 'inline';
            } else {
                console.log(`❌ Compteur non trouvé: ${id}`);
            }
        });
        
        // Mettre à jour le header aussi
        CartManager.updateCartCounter();
    },
    
    /**
     * Initialiser la sélection des formats
     */
    initFormatSelection(image) {
        console.log('🎨 Initialisation OrderManager pour l\'image:', image);
        this.currentImage = image;
        this.formatQuantities = {};
        
        // Initialiser les quantités à 0
        Object.keys(CONFIG.FORMATS).forEach(format => {
            this.formatQuantities[format] = 0;
        });
        
        this.updateSummary();
    },
    
    /**
     * Incrémenter la quantité d'un format
     */
    incrementFormatQuantity(format) {
        const input = document.getElementById(`qty-order-${format}`);
        if (!input) return;
        
        const currentValue = parseInt(input.value) || 0;
        const newValue = currentValue + 1;
        
        input.value = newValue;
        this.updateFormatQuantity(format, newValue);
    },
    
    /**
     * Décrémenter la quantité d'un format
     */
    decrementFormatQuantity(format) {
        const input = document.getElementById(`qty-order-${format}`);
        if (!input) return;
        
        const currentValue = parseInt(input.value) || 0;
        const newValue = Math.max(0, currentValue - 1);
        
        input.value = newValue;
        this.updateFormatQuantity(format, newValue);
    },
    
    /**
     * Mettre à jour la quantité d'un format
     */
    updateFormatQuantity(format, value) {
        const qty = parseInt(value) || 0;
        const validQty = Math.max(0, Math.min(1000, qty));
        
        // Mettre à jour la valeur dans le stockage
        this.formatQuantities[format] = validQty;
        
        // Mettre à jour l'input (ancienne interface)
        const input = document.getElementById(`qty-order-${format}`);
        if (input) {
            input.value = validQty;
        }
        
        // Mettre à jour l'input (nouvelle interface minimaliste)
        const inputMin = document.getElementById(`qty-min-${format}`);
        if (inputMin) {
            inputMin.value = validQty;
        }
        
        // Activer/désactiver le bouton d'ajout pour ce format (ancienne interface)
        const btn = document.getElementById(`btn-add-order-${format}`);
        if (btn) {
            btn.disabled = validQty === 0;
            if (validQty > 0) {
                btn.classList.add('active');
            } else {
                btn.classList.remove('active');
            }
        }
        
        // Activer/désactiver le bouton d'ajout (nouvelle interface minimaliste)
        const btnMin = document.getElementById(`btn-min-${format}`);
        if (btnMin) {
            btnMin.disabled = validQty === 0;
        }
        
        // Mettre à jour le résumé (si présent)
        if (typeof this.updateSummary === 'function') {
            this.updateSummary();
        }
    },
    
    /**
     * Mettre à jour le résumé
     */
    updateSummary() {
        let totalItems = 0;
        let totalPrice = 0;
        
        Object.keys(this.formatQuantities).forEach(format => {
            const qty = this.formatQuantities[format];
            if (qty > 0) {
                totalItems += qty;
                const pricing = PricingManager.calculate(qty, 'economique', format);
                totalPrice += pricing.total;
            }
        });
        
        // Mettre à jour le texte
        const textElement = document.getElementById('total-items-text');
        if (textElement) {
            if (totalItems === 0) {
                textElement.textContent = 'Aucun article sélectionné';
                textElement.style.color = '#94a3b8';
            } else if (totalItems === 1) {
                textElement.textContent = '1 article sélectionné';
                textElement.style.color = '#4ade80';
            } else {
                textElement.textContent = `${totalItems} articles sélectionnés`;
                textElement.style.color = '#4ade80';
            }
        }
        
        // Afficher/masquer le prix total
        const priceDisplay = document.getElementById('total-price-display');
        const priceAmount = document.getElementById('total-price-amount');
        if (priceDisplay && priceAmount) {
            if (totalItems > 0) {
                priceDisplay.style.display = 'block';
                priceAmount.textContent = Utils.formatPrice(totalPrice);
            } else {
                priceDisplay.style.display = 'none';
            }
        }
        
        // Activer/désactiver le bouton "Tout ajouter"
        const btnAddAll = document.getElementById('btn-add-all-formats');
        if (btnAddAll) {
            btnAddAll.disabled = totalItems === 0;
        }
    },
    
    /**
     * Mettre à jour la quantité (version simplifiée avec highlight)
     */
    updateFormatQuantitySimple(format, value) {
        const qty = parseInt(value) || 0;
        const validQty = Math.max(0, Math.min(999, qty));
        
        this.formatQuantities[format] = validQty;
        
        const input = document.getElementById(`qty-${format}`);
        if (input) {
            input.value = validQty === 0 ? '' : validQty;
        }
        
        // Mettre en surbrillance la ligne si quantité > 0
        const row = document.getElementById(`row-${format}`);
        if (row) {
            if (validQty > 0) {
                row.classList.add('selected');
            } else {
                row.classList.remove('selected');
            }
        }
        
        // Activer/désactiver le bouton général
        this.updateAddButton();
    },
    
    /**
     * Mettre à jour l'état du bouton général
     */
    updateAddButton() {
        const hasSelection = Object.values(this.formatQuantities).some(qty => qty > 0);
        const btn = document.getElementById('btn-add-all');
        if (btn) {
            btn.disabled = !hasSelection;
        }
    },
    
    /**
     * Ajouter tous les formats sélectionnés au panier
     */
    addAllSelectedFormats() {
        const state = StateManager.getState();
        let totalAdded = 0;
        let items = [];
        
        Object.keys(this.formatQuantities).forEach(format => {
            const qty = this.formatQuantities[format];
            
            if (qty > 0) {
                const pricing = PricingManager.calculate(qty, 'economique', format);
                
                CartManager.addToCart({
                    generatedImage: this.currentImage.url,
                    selectedStyle: state.selectedStyle,
                    format: format,
                    quantity: qty,
                    deliveryMode: 'economique',
                    pricing: pricing,
                    version: this.currentImage.version
                }, false);
                
                totalAdded += qty;
                items.push(`${qty}× ${format}`);
                
                // Réinitialiser
                this.updateFormatQuantitySimple(format, 0);
            }
        });
        
        if (totalAdded > 0) {
            Utils.showToast(`✅ ${items.join(', ')} ajouté${totalAdded > 1 ? 's' : ''} au panier`);
            
            // Mettre à jour tous les compteurs avec un léger délai pour assurer le rendu
            setTimeout(() => {
                this.updateAllCartCounters();
            }, 50);
            
            // Désactiver le bouton
            this.updateAddButton();
        }
    },
    
    /**
     * Ajouter un format unique au panier (version minimaliste)
     */
    addSingleFormatToCartMinimal(format) {
        const state = StateManager.getState();
        const input = document.getElementById(`qty-min-${format}`);
        const qty = parseInt(input.value) || 0;
        
        if (qty === 0) {
            Utils.showToast('⚠️ Quantité requise');
            input.focus();
            return;
        }
        
        // Calculer le prix
        const pricing = PricingManager.calculate(qty, 'economique', format);
        
        // Ajouter au panier
        CartManager.addToCart({
            generatedImage: this.currentImage.url,
            selectedStyle: state.selectedStyle,
            format: format,
            quantity: qty,
            deliveryMode: 'economique',
            pricing: pricing,
            version: this.currentImage.version
        }, false);
        
        Utils.showToast(`✅ ${qty}× ${format} ajouté`);
        
        // Réinitialiser
        input.value = 0;
        this.updateFormatQuantityMinimal(format, 0);
        
        // Mettre à jour tous les compteurs avec délai
        setTimeout(() => {
            this.updateAllCartCounters();
        }, 50);
    },
    
    /**
     * Mettre à jour la quantité (version minimaliste)
     */
    updateFormatQuantityMinimal(format, value) {
        const qty = parseInt(value) || 0;
        const validQty = Math.max(0, Math.min(999, qty));
        
        this.formatQuantities[format] = validQty;
        
        const input = document.getElementById(`qty-min-${format}`);
        if (input) {
            input.value = validQty;
        }
        
        const btn = document.getElementById(`btn-min-${format}`);
        if (btn) {
            btn.disabled = validQty === 0;
        }
    },
    
    /**
     * Ajouter un format unique au panier
     */
    addSingleFormatToCart(format) {
        const state = StateManager.getState();
        const qty = this.formatQuantities[format] || 0;
        
        if (qty === 0) {
            Utils.showToast('⚠️ Veuillez sélectionner une quantité');
            return;
        }
        
        // Calculer le prix
        const pricing = PricingManager.calculate(qty, 'economique', format);
        
        // Ajouter au panier
        CartManager.addToCart({
            generatedImage: this.currentImage.url,
            selectedStyle: state.selectedStyle,
            format: format,
            quantity: qty,
            deliveryMode: 'economique',
            pricing: pricing,
            version: this.currentImage.version
        }, false);
        
        Utils.showToast(`✅ ${qty}× ${format} ajouté${qty > 1 ? 's' : ''} au panier`);
        
        // Réinitialiser la quantité
        this.updateFormatQuantity(format, 0);
        
        // Mettre à jour tous les compteurs avec délai
        setTimeout(() => {
            this.updateAllCartCounters();
        }, 50);
    },
    
    /**
     * Ajouter tous les formats au panier
     */
    addAllFormatsToCart() {
        const state = StateManager.getState();
        let totalAdded = 0;
        
        Object.keys(this.formatQuantities).forEach(format => {
            const qty = this.formatQuantities[format];
            
            if (qty > 0) {
                const pricing = PricingManager.calculate(qty, 'economique', format);
                
                CartManager.addToCart({
                    generatedImage: this.currentImage.url,
                    selectedStyle: state.selectedStyle,
                    format: format,
                    quantity: qty,
                    deliveryMode: 'economique',
                    pricing: pricing,
                    version: this.currentImage.version
                }, false);
                
                totalAdded += qty;
                
                // Réinitialiser la quantité
                this.updateFormatQuantity(format, 0);
            }
        });
        
        if (totalAdded > 0) {
            Utils.showToast(`✅ ${totalAdded} article${totalAdded > 1 ? 's' : ''} ajouté${totalAdded > 1 ? 's' : ''} au panier`);
            
            // Mettre à jour tous les compteurs avec délai
            setTimeout(() => {
                this.updateAllCartCounters();
            }, 50);
        }
    },
    
    /**
     * Sélectionne un format
     */
    selectFormat(format) {
        StateManager.setState({ format });
        RenderManager.render();
    },
    
    /**
     * Sélectionne un mode de livraison
     */
    selectDelivery(mode) {
        StateManager.setState({ deliveryMode: mode });
        RenderManager.render();
    },
    
    /**
     * Met à jour la quantité
     */
    updateQuantity(event) {
        const input = event.target;
        let qty = parseInt(input.value);
        
        if (input.value === '' || input.value === '0') {
            return;
        }
        
        if (isNaN(qty) || qty < 1) qty = 1;
        if (qty > 1000) qty = 1000;
        
        StateManager.setState({ quantity: qty });
        PricingManager.updateDisplay();
    },
    
    /**
     * Sélectionne une quantité depuis la grille
     */
    selectQuantityFromGrid(qty) {
        document.getElementById('quantity').value = qty;
        StateManager.setState({ quantity: qty });
        RenderManager.render();
    },
    
    /**
     * Soumet une commande (produit unique)
     */
    async submitOrder(event) {
        event.preventDefault();
        
        const state = StateManager.getState();
        
        if (state.isSubmitting) return;
        
        StateManager.setState({ isSubmitting: true });
        
        const submitButton = event.target.querySelector('button[type="submit"]');
        const buttonContent = submitButton?.querySelector('.btn-content');
        
        if (submitButton) {
            submitButton.disabled = true;
            submitButton.classList.add('loading');
        }
        
        if (buttonContent) {
            buttonContent.innerHTML = '<div class="spinner"></div> Traitement en cours...';
        }
        
        try {
            const pricing = PricingManager.calculate(
                state.quantity, 
                state.deliveryMode, 
                state.format
            );
            
            // Ouvrir la modale de paiement
            const paymentResult = await openPaymentModal(pricing.total);
            
            if (!paymentResult.success) {
                throw new Error(paymentResult.message || 'Paiement annulé');
            }
            
            // Préparer les données de commande
            const fd = new FormData();
            
            // Informations client
            Object.keys(state.customer).forEach(key => {
                fd.append(key, state.customer[key]);
            });
            
            // Informations commande
            fd.append('image_url', state.generatedImage);
            fd.append('style', state.selectedStyle.titre);
            fd.append('format', CONFIG.FORMATS[state.format].label);
            fd.append('quantity', state.quantity);
            fd.append('delivery_mode', CONFIG.DELIVERY_LABELS[state.deliveryMode].name);
            fd.append('total_price', pricing.total.toFixed(2));
            fd.append('session_id', state.sessionId);
            
            // Informations authentification
            if (authManager && authManager.isAuthenticated) {
                fd.append('user_id', authManager.user.id);
                fd.append('user_email', authManager.user.email);
                fd.append('user_authenticated', 'true');
            }
            
            // Informations paiement
            fd.append('payment_id', paymentResult.payment_id);
            fd.append('payment_method', 'stripe');
            fd.append('payment_status', 'paid');
            
            // Envoyer la commande
            await ApiManager.submitOrder(fd);
            
            // Succès
            StateManager.setState({
                view: 'success',
                paymentId: paymentResult.payment_id,
                isSubmitting: false
            });
            
            RenderManager.render();
            Utils.scrollToTop();
            
        } catch (error) {
            console.error('Erreur:', error);
            
            if (error.message !== 'Paiement annulé') {
                Utils.showToast('❌ ' + error.message);
            }
            
            StateManager.setState({ isSubmitting: false });
            
            if (submitButton) {
                submitButton.disabled = false;
                submitButton.classList.remove('loading');
            }
            
            if (buttonContent) {
                const state = StateManager.getState();
                const pricing = PricingManager.calculate(
                    state.quantity, 
                    state.deliveryMode, 
                    state.format
                );
                buttonContent.innerHTML = `💳 Commander - ${Utils.formatPrice(pricing.total)}`;
            }
        }
    },
    
    /**
     * Soumet une commande panier
     */
    async submitCartOrder(event) {
        event.preventDefault();
        
        const state = StateManager.getState();
        
        if (state.cart.length === 0) {
            Utils.showToast('⚠️ Votre panier est vide');
            return;
        }
        
        if (state.isSubmitting) return;
        
        StateManager.setState({ isSubmitting: true });
        
        const submitButton = event.target.querySelector('button[type="submit"]');
        const buttonContent = submitButton?.querySelector('.btn-content');
        
        if (submitButton) {
            submitButton.disabled = true;
            submitButton.classList.add('loading');
        }
        
        if (buttonContent) {
            buttonContent.innerHTML = '<div class="spinner"></div> Traitement en cours...';
        }
        
        try {
            const cartTotal = CartManager.getTotal();
            const totalWithPromo = CartManager.getTotalWithPromo();
            const totalUnits = CartManager.getTotalUnits();
            
            // Utiliser le total avec réduction si code promo appliqué
            const finalTotal = totalWithPromo;
            
            // Ouvrir la modale de paiement avec le bon montant
            const paymentResult = await openPaymentModal(finalTotal);
            
            if (!paymentResult.success) {
                throw new Error(paymentResult.message || 'Paiement annulé');
            }
            
            // Préparer les données
            const fd = new FormData();
            
            // Informations client
            Object.keys(state.customer).forEach(key => {
                fd.append(key, state.customer[key]);
            });
            
            // Informations authentification
            if (authManager && authManager.isAuthenticated) {
                fd.append('user_id', authManager.user.id);
                fd.append('user_email', authManager.user.email);
                fd.append('user_authenticated', 'true');
            }
            
            // Informations paiement
            fd.append('payment_id', paymentResult.payment_id);
            fd.append('payment_method', 'stripe');
            fd.append('payment_status', 'paid');
            
            // Informations commande
            fd.append('order_type', 'cart');
            fd.append('total_items', state.cart.length);
            fd.append('total_quantity', totalUnits);
            fd.append('total_price', finalTotal.toFixed(2));
            
            // Informations code promo si appliqué
            if (CartManager.promoCode) {
                fd.append('promo_code', CartManager.promoCode);
                fd.append('promo_reduction', CartManager.promoReduction.toFixed(2));
                fd.append('total_before_promo', cartTotal.toFixed(2));
            }
            
            // Articles du panier
            state.cart.forEach((item, index) => {
                fd.append(`item_${index}_id`, item.id);
                fd.append(`item_${index}_style`, item.styleName);
                fd.append(`item_${index}_format`, item.formatLabel);
                fd.append(`item_${index}_quantity`, item.quantity);
                fd.append(`item_${index}_delivery`, item.deliveryLabel);
                fd.append(`item_${index}_price`, item.totalPrice.toFixed(2));
                fd.append(`item_${index}_image`, item.image);
            });
            
            // Envoyer la commande
            await ApiManager.submitOrder(fd);
            
            // Vider le panier, réinitialiser le code promo et afficher succès
            CartManager.removePromoCode();
            StateManager.setState({
                cart: [],
                cartCount: 0,
                isSubmitting: false,
                view: 'success',
                paymentId: paymentResult.payment_id
            });
            
            StateManager.saveCartToStorage();
            CartManager.updateCartCounter();
            RenderManager.render();
            Utils.scrollToTop();
            
        } catch (error) {
            console.error('Erreur:', error);
            
            if (error.message !== 'Paiement annulé') {
                Utils.showToast('❌ ' + error.message);
            }
            
            StateManager.setState({ isSubmitting: false });
            
            if (submitButton) {
                submitButton.disabled = false;
                submitButton.classList.remove('loading');
            }
            
            if (buttonContent) {
                const finalTotal = CartManager.getTotalWithPromo();
                buttonContent.innerHTML = `💳 Payer et confirmer ma commande - ${Utils.formatPrice(finalTotal)}`;
            }
        }
    }
};
