/**
 * Gestionnaire de navigation
 */

const NavigationManager = {
    /**
     * Retour à l'accueil
     */
    goHome() {
        StateManager.resetToHome();
        RenderManager.render();
    },
    
    /**
     * Filtre par catégorie
     */
    filterByCategory(categorySlug) {
        const state = StateManager.getState();
        
        let filteredStyles;
        if (categorySlug === 'all') {
            filteredStyles = state.allStyles;
        } else {
            filteredStyles = state.allStyles.filter(
                style => style.categorie_slug === categorySlug
            );
        }
        
        StateManager.setState({
            selectedCategory: categorySlug,
            styles: filteredStyles,
            searchQuery: null,  // Reset la recherche
            showingFavoris: false  // Reset favoris
        });
        
        // Vider le champ de recherche
        const searchInput = document.getElementById('searchInput');
        if (searchInput) searchInput.value = '';
        
        RenderManager.render();
    },
    
    /**
     * Sélectionne un style
     */
    selectStyle(style) {
        console.log('🎨 Style sélectionné:', style);
        
        StateManager.setState({
            selectedStyle: style,
            backgroundType: 'ia',
            imageTitle: '',
            sceneDescription: '',
            uploadedFile: null,
            uploadedPreview: null,
            view: 'result-generate'  // Nouvelle vue
        });
        
        RenderManager.render();
    },
    
    /**
     * Retour aux catégories
     */
    backToCategories() {
        StateManager.setState({
            view: 'categories',
            selectedCategory: null
        });
        RenderManager.render();
    },
    
    /**
     * Retour aux styles
     */
    backToStyles() {
        StateManager.setState({
            view: 'styles'
        });
        RenderManager.render();
    },
    
    /**
     * Validation de l'image générée
     */
    validateImage() {
        document.body.classList.add('order-page');
        
        StateManager.setState({
            view: 'order',
            format: 'A3',
            quantity: 1,
            deliveryMode: 'economique'
        });
        
        RenderManager.render();
    },
    
    /**
     * Commander toutes les versions
     */
    validateAllImages() {
        const state = StateManager.getState();
        
        // Ajouter toutes les versions au panier
        state.generatedImages.forEach((img, index) => {
            const pricing = PricingManager.calculate(1, 'economique', 'A3');
            
            CartManager.addToCart({
                generatedImage: img.url,
                selectedStyle: state.selectedStyle,
                format: 'A3',
                quantity: 1,
                deliveryMode: 'economique',
                pricing: pricing
            }, false); // false = ne pas afficher le toast à chaque ajout
        });
        
        Utils.showToast(`✅ ${state.generatedImages.length} versions ajoutées au panier`);
        
        // Aller au panier
        StateManager.setState({ view: 'cart' });
        RenderManager.render();
    },
    
    /**
     * Regénérer une image (créer une autre version)
     */
    regenerate() {
        StateManager.setState({
            view: 'upload'
        });
        RenderManager.render();
    },
    
    /**
     * Passer à la commande (checkout)
     */
    async proceedToCheckout() {
        const state = StateManager.getState();
        
        if (state.cart.length === 0) {
            Utils.showToast('⚠️ Votre panier est vide');
            return;
        }
        
        console.log('🔍 DEBUG - Début proceedToCheckout');
        console.log('authManager existe:', typeof authManager !== 'undefined');
        
        // Vérifier si authManager existe
        if (typeof authManager === 'undefined') {
            console.warn('⚠️ authManager non disponible');
            // Continuer sans authentification
            StateManager.setState({
                view: 'checkout'
            });
            RenderManager.render();
            return;
        }
        
        console.log('Authentifié:', authManager.isAuthenticated);
        console.log('User:', authManager.user);
        
        // Si non connecté, demander la connexion
        if (!authManager.isAuthenticated) {
            console.log('❌ Non connecté - Affichage modale');
            
            // Vérifier si showAuthModal existe
            if (typeof showAuthModal === 'function') {
                showAuthModal('login', () => {
                    console.log('✅ Connexion réussie - Remplissage des données');
                    const customer = authManager.fillCustomerData(state.customer);
                    StateManager.setState({
                        customer: customer,
                        view: 'checkout'
                    });
                    RenderManager.render();
                });
            } else {
                console.warn('⚠️ showAuthModal non disponible');
                // Continuer sans authentification
                StateManager.setState({
                    view: 'checkout'
                });
                RenderManager.render();
            }
            return;
        }
        
        // Si déjà connecté, pré-remplir les données
        console.log('✅ Déjà connecté - Remplissage des données');
        const customer = authManager.fillCustomerData(state.customer);
        
        StateManager.setState({
            customer: customer,
            view: 'checkout'
        });
        
        RenderManager.render();
    },
    
    /**
     * Afficher le panier
     */
    viewCart() {
        console.log('🛒 Ouverture du panier');
        
        // Vérifier si authManager existe et est chargé
        if (typeof authManager === 'undefined') {
            console.warn('⚠️ authManager non disponible');
            // Afficher le panier quand même
            StateManager.setState({ view: 'cart' });
            RenderManager.render();
            return;
        }
        
        console.log('Auth status:', authManager.isAuthenticated);
        
        // Si non connecté, demander la connexion
        if (!authManager.isAuthenticated) {
            console.log('❌ Non connecté - Affichage modale');
            
            // Vérifier si showAuthModal existe
            if (typeof showAuthModal === 'function') {
                showAuthModal('login', () => {
                    console.log('✅ Connexion réussie - Affichage du panier');
                    StateManager.setState({ view: 'cart' });
                    RenderManager.render();
                });
            } else {
                console.warn('⚠️ showAuthModal non disponible');
                alert('Veuillez vous connecter pour accéder à votre panier');
            }
            return;
        }
        
        // Si connecté, afficher le panier
        console.log('✅ Connecté - Affichage du panier');
        StateManager.setState({ view: 'cart' });
        RenderManager.render();
    },
    
    /**
     * Continuer les achats depuis le panier
     */
    continueShopping() {
        this.goHome();
    },
    
    /**
     * Retour à la page de résultat avec les versions générées
     */
    backToResult() {
        const state = StateManager.getState();
        
        // Si on a des images générées, retourner à la page avec les 3 versions
        if (state.generatedImages && state.generatedImages.length > 0) {
            StateManager.setState({
                view: 'result-generate',
                selectedImagesForFormats: null
            });
        } else {
            StateManager.setState({
                view: 'result',
                selectedImagesForFormats: null
            });
        }
        RenderManager.render();
    },
    
    /**
     * Retour à la sélection des formats depuis le panier
     */
    backToFormatSelection() {
        const state = StateManager.getState();
        
        // S'il y a des images générées, aller à la sélection des formats
        if (state.generatedImages && state.generatedImages.length > 0) {
            // Préparer les images pour la sélection de format
            const imagesForFormats = state.generatedImages.map((img, idx) => ({
                image: img,
                index: idx
            }));
            
            StateManager.setState({
                view: 'format-selection',
                selectedImagesForFormats: imagesForFormats
            });
        } else {
            // Sinon retour à l'accueil
            StateManager.resetToHome();
        }
        RenderManager.render();
    }
};