/**
 * MODULE D'AUTHENTIFICATION - Ça Cartoon
 * Gestion complète de l'inscription, connexion et profil client
 */

const AUTH_API_URL = 'auth.php';
const TOKEN_KEY = 'cacartoon_token';
const USER_KEY = 'cacartoon_user';

// =====================================================
// CLASSE AUTHMANAGER - GESTION CENTRALISÉE
// =====================================================
class AuthManager {
    constructor() {
        this.token = localStorage.getItem(TOKEN_KEY);
        this.user = this.loadUser();
        this.isAuthenticated = this.token && this.user;
        
        // Vérifier le token au démarrage
        if (this.token) {
            this.verifyToken();
        }
    }
    
    // Charger l'utilisateur depuis le localStorage
    loadUser() {
        try {
            const userData = localStorage.getItem(USER_KEY);
            return userData ? JSON.parse(userData) : null;
        } catch (e) {
            console.error('Erreur chargement utilisateur:', e);
            return null;
        }
    }
    
    // Sauvegarder l'utilisateur et le token
    saveAuth(token, user) {
        this.token = token;
        this.user = user;
        this.isAuthenticated = true;
        
        localStorage.setItem(TOKEN_KEY, token);
        localStorage.setItem(USER_KEY, JSON.stringify(user));
        
        console.log('✅ Authentification sauvegardée:', user.email);
    }
    
    // Déconnexion
    logout() {
        this.token = null;
        this.user = null;
        this.isAuthenticated = false;
        
        localStorage.removeItem(TOKEN_KEY);
        localStorage.removeItem(USER_KEY);
        
        console.log('👋 Déconnexion effectuée');
    }
    
    // Vérifier la validité du token
    async verifyToken() {
        try {
            const response = await fetch(`${AUTH_API_URL}?action=verify-token`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ token: this.token })
            });
            
            const data = await response.json();
            
            if (!data.success) {
                console.warn('⚠️ Token invalide, déconnexion');
                this.logout();
                return false;
            }
            
            return true;
        } catch (e) {
            console.error('Erreur vérification token:', e);
            return false;
        }
    }
    
    // Inscription
    async register(email, password, nom, prenom, telephone, adresse, ville, code_postal) {
        try {
            const response = await fetch(`${AUTH_API_URL}?action=register`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ email, password, nom, prenom, telephone, adresse, ville, code_postal })
            });
            
            const data = await response.json();
            
            if (!data.success) {
                throw new Error(data.error || 'Erreur lors de l\'inscription');
            }
            
            this.saveAuth(data.data.token, data.data.user);
            return data;
            
        } catch (e) {
            console.error('Erreur inscription:', e);
            throw e;
        }
    }
    
    // Connexion
    async login(email, password) {
        try {
            const response = await fetch(`${AUTH_API_URL}?action=login`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ email, password })
            });
            
            const data = await response.json();
            
            if (!data.success) {
                throw new Error(data.error || 'Erreur lors de la connexion');
            }
            
            this.saveAuth(data.data.token, data.data.user);
            return data;
            
        } catch (e) {
            console.error('Erreur connexion:', e);
            throw e;
        }
    }
    
    // Récupérer le profil complet
    async getProfile() {
        try {
            const response = await fetch(`${AUTH_API_URL}?action=profile`, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.token}`
                }
            });
            
            const data = await response.json();
            
            if (!data.success) {
                throw new Error(data.error);
            }
            
            // Mettre à jour les données locales
            this.user = data.data;
            localStorage.setItem(USER_KEY, JSON.stringify(this.user));
            
            return data.data;
            
        } catch (e) {
            console.error('Erreur récupération profil:', e);
            throw e;
        }
    }
    
    // Mettre à jour le profil
    async updateProfile(fields) {
        try {
            // Ajouter le token dans les données envoyées (en backup si header ne passe pas)
            const dataToSend = {
                ...fields,
                token: this.token
            };
            
            const response = await fetch(`${AUTH_API_URL}?action=update`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.token}`
                },
                body: JSON.stringify(dataToSend)
            });
            
            const data = await response.json();
            
            if (!data.success) {
                throw new Error(data.error);
            }
            
            // Mettre à jour les données locales
            this.user = data.data;
            localStorage.setItem(USER_KEY, JSON.stringify(this.user));
            
            return data.data;
            
        } catch (e) {
            console.error('Erreur mise à jour profil:', e);
            throw e;
        }
    }
    
    // Changer le mot de passe
    async changePassword(currentPassword, newPassword) {
        try {
            const response = await fetch(`${AUTH_API_URL}?action=change-password`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.token}`
                },
                body: JSON.stringify({
                    current_password: currentPassword,
                    new_password: newPassword
                })
            });
            
            const data = await response.json();
            
            if (!data.success) {
                throw new Error(data.error);
            }
            
            return data;
            
        } catch (e) {
            console.error('Erreur changement mot de passe:', e);
            throw e;
        }
    }
    
    // Pré-remplir les données client dans le state
    fillCustomerData(stateCustomer) {
        if (!this.isAuthenticated || !this.user) {
            return stateCustomer;
        }
        
        return {
            ...stateCustomer,
            name: this.user.nom || '',
            prenom: this.user.prenom || '',
            email: this.user.email,
            phone: this.user.telephone || '',
            address: this.user.adresse || '',
            city: this.user.ville || '',
            zip: this.user.code_postal || ''
        };
    }
}

// =====================================================
// INTERFACE UTILISATEUR - MODALES
// =====================================================

// Afficher la modale de connexion/inscription
function showAuthModal(mode = 'login', onSuccess = null) {
    const existingModal = document.getElementById('authModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    const modalHTML = `
        <div id="authModal" class="auth-modal-overlay">
            <div class="auth-modal">
                <button class="auth-modal-close" onclick="closeAuthModal()">✕</button>
                
                <div class="auth-modal-header">
                    <h2 id="authModalTitle">${mode === 'login' ? 'Connexion' : 'Inscription'}</h2>
                    <p id="authModalSubtitle">
                        ${mode === 'login' 
                            ? 'Connectez-vous pour accéder à votre compte' 
                            : 'Créez votre compte pour commander'}
                    </p>
                </div>
                
                <div id="authFormContainer">
                    ${mode === 'login' ? getLoginFormHTML() : getRegisterFormHTML()}
                </div>
                
                <div class="auth-modal-footer">
                    <button type="button" class="auth-switch-btn" onclick="toggleAuthMode()">
                        ${mode === 'login' 
                            ? 'Pas encore de compte ? Inscrivez-vous' 
                            : 'Déjà un compte ? Connectez-vous'}
                    </button>
                </div>
            </div>
        </div>
    `;
    
    document.body.insertAdjacentHTML('beforeend', modalHTML);
    
    // Ajouter le gestionnaire de clic APRÈS l'insertion
    const overlay = document.getElementById('authModal');
    const modal = overlay.querySelector('.auth-modal');
    
    let mouseDownTarget = null;
    
    // Enregistrer où le clic a commencé
    overlay.addEventListener('mousedown', (e) => {
        mouseDownTarget = e.target;
    });
    
    // Ne fermer que si le clic commence ET finit sur l'overlay (pas sur le contenu)
    overlay.addEventListener('click', (e) => {
        // Si le mousedown et le click sont tous les deux sur l'overlay (pas sur le contenu)
        if (e.target === overlay && mouseDownTarget === overlay) {
            closeAuthModal();
        }
    });
    
    // Empêcher la propagation des clics à l'intérieur de la modale
    modal.addEventListener('click', (e) => {
        e.stopPropagation();
    });
    
    // Stocker le callback de succès
    window.authModalSuccessCallback = onSuccess;
    
    // Gestionnaire de la touche Échap
    const handleEscape = (e) => {
        if (e.key === 'Escape') {
            closeAuthModal();
            document.removeEventListener('keydown', handleEscape);
        }
    };
    document.addEventListener('keydown', handleEscape);
    
    // Focus sur le premier champ
    setTimeout(() => {
        const firstInput = document.querySelector('#authModal input');
        if (firstInput) firstInput.focus();
    }, 100);
}

// HTML du formulaire de connexion
function getLoginFormHTML() {
    return `
        <form id="loginForm" onsubmit="handleLogin(event)">
            <div class="auth-form-group">
                <label>Email</label>
                <input type="email" name="email" required placeholder="votre@email.fr" />
            </div>
            
            <div class="auth-form-group">
                <label>Mot de passe</label>
                <input type="password" name="password" required placeholder="••••••••" />
            </div>
            
            <div id="authError" class="auth-error" style="display: none;"></div>
            
            <button type="submit" class="auth-btn-primary" id="loginBtn">
                Se connecter
            </button>
        </form>
    `;
}

// HTML du formulaire d'inscription
function getRegisterFormHTML() {
    return `
        <form id="registerForm" onsubmit="handleRegister(event)">
            <div class="auth-form-row">
                <div class="auth-form-group">
                    <label>Nom *</label>
                    <input type="text" name="nom" required placeholder="Dupont" />
                </div>
                
                <div class="auth-form-group">
                    <label>Prénom</label>
                    <input type="text" name="prenom" placeholder="Jean" />
                </div>
            </div>
            
            <div class="auth-form-group">
                <label>Email *</label>
                <input type="email" name="email" required placeholder="votre@email.fr" />
            </div>
            
            <div class="auth-form-group">
                <label>Téléphone *</label>
                <input type="tel" name="telephone" required placeholder="06 12 34 56 78" />
            </div>
            
            <div class="auth-form-group">
                <label>Adresse</label>
                <input type="text" name="adresse" placeholder="123 Rue de la Paix" />
            </div>
            
            <div class="auth-form-row">
                <div class="auth-form-group">
                    <label>Ville</label>
                    <input type="text" name="ville" placeholder="Paris" />
                </div>
                
                <div class="auth-form-group">
                    <label>Code postal</label>
                    <input type="text" name="code_postal" placeholder="75001" />
                </div>
            </div>
            
            <div class="auth-form-group">
                <label>Mot de passe *</label>
                <input type="password" name="password" required placeholder="••••••••" />
                <small class="auth-password-hint">
                    Min. 8 caractères, 1 majuscule, 1 minuscule, 1 chiffre
                </small>
            </div>
            
            <div class="auth-form-group">
                <label>Confirmer le mot de passe *</label>
                <input type="password" name="password_confirm" required placeholder="••••••••" />
            </div>
            
            <div id="authError" class="auth-error" style="display: none;"></div>
            
            <button type="submit" class="auth-btn-primary" id="registerBtn">
                Créer mon compte
            </button>
        </form>
    `;
}

// Basculer entre connexion et inscription
function toggleAuthMode() {
    const modal = document.getElementById('authModal');
    if (!modal) return;
    
    const title = document.getElementById('authModalTitle');
    const subtitle = document.getElementById('authModalSubtitle');
    const container = document.getElementById('authFormContainer');
    const switchBtn = document.querySelector('.auth-switch-btn');
    
    const isLogin = title.textContent === 'Connexion';
    
    if (isLogin) {
        title.textContent = 'Inscription';
        subtitle.textContent = 'Créez votre compte pour commander';
        container.innerHTML = getRegisterFormHTML();
        switchBtn.textContent = 'Déjà un compte ? Connectez-vous';
    } else {
        title.textContent = 'Connexion';
        subtitle.textContent = 'Connectez-vous pour accéder à votre compte';
        container.innerHTML = getLoginFormHTML();
        switchBtn.textContent = 'Pas encore de compte ? Inscrivez-vous';
    }
    
    // Focus sur le premier champ
    setTimeout(() => {
        const firstInput = container.querySelector('input');
        if (firstInput) firstInput.focus();
    }, 100);
}

// Fermer la modale
function closeAuthModal() {
    const modal = document.getElementById('authModal');
    if (modal) {
        modal.classList.add('closing');
        setTimeout(() => modal.remove(), 300);
    }
    
    window.authModalSuccessCallback = null;
}

// Afficher une erreur
function showAuthError(message) {
    const errorDiv = document.getElementById('authError');
    if (errorDiv) {
        errorDiv.textContent = message;
        errorDiv.style.display = 'block';
    }
}

// Gérer la connexion
async function handleLogin(event) {
    event.preventDefault();
    
    const btn = document.getElementById('loginBtn');
    const form = event.target;
    const formData = new FormData(form);
    
    const email = formData.get('email');
    const password = formData.get('password');
    
    btn.disabled = true;
    btn.innerHTML = '<div class="spinner"></div> Connexion...';
    
    try {
        await authManager.login(email, password);
        
        closeAuthModal();
        
        // Charger les favoris du client
        if (typeof FavorisManager !== 'undefined') {
            await FavorisManager.init();
            console.log('❤️ Favoris chargés après connexion');
        }
        
        // Callback de succès
        if (window.authModalSuccessCallback) {
            window.authModalSuccessCallback();
        } else {
            // Par défaut, mettre à jour l'UI
            updateAuthUI();
        }
        
    } catch (error) {
        showAuthError(error.message);
        btn.disabled = false;
        btn.textContent = 'Se connecter';
    }
}

// Gérer l'inscription
async function handleRegister(event) {
    event.preventDefault();
    
    const btn = document.getElementById('registerBtn');
    const form = event.target;
    const formData = new FormData(form);
    
    const email = formData.get('email');
    const password = formData.get('password');
    const passwordConfirm = formData.get('password_confirm');
    const nom = formData.get('nom');
    const prenom = formData.get('prenom');
    const telephone = formData.get('telephone');
    const adresse = formData.get('adresse');
    const ville = formData.get('ville');
    const code_postal = formData.get('code_postal');
    
    // DEBUG - Afficher les valeurs récupérées
    console.log('🔍 DEBUG INSCRIPTION - Valeurs du formulaire:');
    console.log('  email:', email);
    console.log('  nom:', nom);
    console.log('  prenom:', prenom);
    console.log('  telephone:', telephone);
    console.log('  adresse:', `"${adresse}"`, '(longueur:', adresse?.length || 0, ')');
    console.log('  ville:', `"${ville}"`, '(longueur:', ville?.length || 0, ')');
    console.log('  code_postal:', `"${code_postal}"`, '(longueur:', code_postal?.length || 0, ')');
    
    // Vérifier que les mots de passe correspondent
    if (password !== passwordConfirm) {
        showAuthError('Les mots de passe ne correspondent pas');
        return;
    }
    
    btn.disabled = true;
    btn.innerHTML = '<div class="spinner"></div> Création du compte...';
    
    try {
        await authManager.register(email, password, nom, prenom, telephone, adresse, ville, code_postal);
        
        showToast('✅ Compte créé avec succès !');
        closeAuthModal();
        
        // Initialiser les favoris (nouvel utilisateur, donc vide mais prêt)
        if (typeof FavorisManager !== 'undefined') {
            await FavorisManager.init();
        }
        
        // Callback de succès
        if (window.authModalSuccessCallback) {
            window.authModalSuccessCallback();
        } else {
            updateAuthUI();
        }
        
    } catch (error) {
        showAuthError(error.message);
        btn.disabled = false;
        btn.textContent = 'Créer mon compte';
    }
}

// Mettre à jour l'interface selon l'état d'authentification
function updateAuthUI() {
    const authButton = document.getElementById('authButton');
    
    if (!authButton) return;
    
    // Supprimer l'ancien menu dropdown s'il existe
    const existingDropdown = document.getElementById('userDropdownMenu');
    if (existingDropdown) {
        existingDropdown.remove();
    }
    
    if (authManager.isAuthenticated) {
        // Créer un conteneur pour le bouton et le menu
        authButton.classList.add('user-menu-container');
        authButton.innerHTML = `
            <span class="action-icon">👤</span>
            <span class="action-text">${authManager.user.prenom || 'Mon compte'}</span>
            <span class="dropdown-arrow">▼</span>
            <div class="user-dropdown-menu" id="userDropdownMenu">
                <div class="user-dropdown-header">
                    <span class="user-greeting">Bonjour, ${authManager.user.prenom || authManager.user.nom || 'Client'} !</span>
                    <span class="user-email">${authManager.user.email}</span>
                </div>
                <div class="user-dropdown-divider"></div>
                <a href="#" class="user-dropdown-item" onclick="showAccountSettings(); return false;">
                    <span class="dropdown-icon">⚙️</span>
                    Gérer mon compte
                </a>
                <a href="#" class="user-dropdown-item" onclick="showOrderHistory(); return false;">
                    <span class="dropdown-icon">📦</span>
                    Mes commandes
                </a>
                <div class="user-dropdown-divider"></div>
                <a href="#" class="user-dropdown-item logout-item" onclick="handleLogout(); return false;">
                    <span class="dropdown-icon">🚪</span>
                    Se déconnecter
                </a>
            </div>
        `;
        
        // Toggle du menu au clic
        authButton.onclick = (e) => {
            e.preventDefault();
            e.stopPropagation();
            authButton.classList.toggle('menu-open');
        };
        
        // Fermer le menu si on clique ailleurs
        document.addEventListener('click', closeUserMenu);
        
    } else {
        // Afficher "Se connecter"
        authButton.classList.remove('user-menu-container', 'menu-open');
        authButton.innerHTML = `
            <span class="action-icon">👤</span>
            <span class="action-text">Se connecter</span>
        `;
        authButton.onclick = (e) => {
            e.preventDefault();
            showAuthModal('login');
        };
        
        // Retirer l'écouteur
        document.removeEventListener('click', closeUserMenu);
    }
}

// Fermer le menu utilisateur
function closeUserMenu(e) {
    const authButton = document.getElementById('authButton');
    if (authButton && !authButton.contains(e.target)) {
        authButton.classList.remove('menu-open');
    }
}

// Afficher l'historique des commandes
async function showOrderHistory() {
    closeUserMenuManually();
    
    // Supprimer une éventuelle modale existante
    const existingModal = document.getElementById('orderHistoryModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    const modalHTML = `
        <div id="orderHistoryModal" class="auth-modal-overlay">
            <div class="auth-modal order-history-modal">
                <button class="auth-modal-close" onclick="closeOrderHistory()">✕</button>
                
                <div class="auth-modal-header compact-header">
                    <h2>📦 Mes commandes</h2>
                </div>
                
                <div id="orderHistoryContent" class="order-history-content">
                    <div class="orders-loading">
                        <div class="spinner"></div>
                        <span>Chargement de vos commandes...</span>
                    </div>
                </div>
                
                <div class="modal-footer-actions">
                    <button type="button" class="btn-close-modal" onclick="closeOrderHistory()">
                        ✕ Fermer
                    </button>
                </div>
            </div>
        </div>
    `;
    
    document.body.insertAdjacentHTML('beforeend', modalHTML);
    
    // Gestionnaire de fermeture
    const overlay = document.getElementById('orderHistoryModal');
    const modal = overlay.querySelector('.auth-modal');
    
    let mouseDownTarget = null;
    overlay.addEventListener('mousedown', (e) => mouseDownTarget = e.target);
    overlay.addEventListener('click', (e) => {
        if (e.target === overlay && mouseDownTarget === overlay) closeOrderHistory();
    });
    modal.addEventListener('click', (e) => e.stopPropagation());
    
    // Touche Échap
    const handleEscape = (e) => {
        if (e.key === 'Escape') {
            closeOrderHistory();
            document.removeEventListener('keydown', handleEscape);
        }
    };
    document.addEventListener('keydown', handleEscape);
    
    // Charger les commandes
    await loadOrderHistory();
}

// Fermer la modale des commandes
function closeOrderHistory() {
    const modal = document.getElementById('orderHistoryModal');
    if (modal) {
        modal.classList.add('closing');
        setTimeout(() => modal.remove(), 300);
    }
}

// Charger l'historique des commandes depuis l'API
async function loadOrderHistory() {
    const container = document.getElementById('orderHistoryContent');
    
    try {
        const clientId = authManager.user?.id;
        if (!clientId) {
            throw new Error('Utilisateur non connecté');
        }
        
        const response = await fetch(`${CONFIG.API_URL}?action=client_orders&client_id=${clientId}`);
        const data = await response.json();
        
        if (!data.success) {
            throw new Error(data.error || 'Erreur lors du chargement');
        }
        
        if (!data.data || data.data.length === 0) {
            container.innerHTML = `
                <div class="no-orders">
                    <span class="no-orders-icon">📭</span>
                    <p>Vous n'avez pas encore passé de commande</p>
                    <button class="btn-start-shopping" onclick="closeOrderHistory(); goHome();">
                        🛍️ Commencer mes achats
                    </button>
                </div>
            `;
            return;
        }
        
        // Afficher les commandes
        container.innerHTML = data.data.map(order => renderOrderCard(order)).join('');
        
    } catch (error) {
        console.error('Erreur chargement commandes:', error);
        container.innerHTML = `
            <div class="orders-error">
                <span>❌</span>
                <p>Impossible de charger vos commandes</p>
                <button onclick="loadOrderHistory()">Réessayer</button>
            </div>
        `;
    }
}

// Rendu d'une carte de commande
function renderOrderCard(order) {
    const date = new Date(order.date_commande).toLocaleDateString('fr-FR', {
        day: 'numeric', month: 'long', year: 'numeric'
    });
    
    const statusInfo = getOrderStatus(order);
    const nbArticles = order.nb_articles || 1;
    const montant = parseFloat(order.montant_total || 0).toFixed(2);
    
    return `
        <div class="order-card" id="order-card-${order.id}">
            <div class="order-status-badge ${statusInfo.class}">
                ${statusInfo.label}
            </div>
            
            <div class="order-card-header">
                <div class="order-info">
                    <span class="order-number">Commande #${order.id}</span>
                    <span class="order-date">${date}</span>
                </div>
                <div class="order-summary">
                    <span class="order-items">${nbArticles} article${nbArticles > 1 ? 's' : ''}</span>
                    <span class="order-total">${montant} €</span>
                </div>
            </div>
            
            <div class="order-tracking">
                <div class="tracking-steps">
                    <div class="tracking-step ${statusInfo.step >= 1 ? 'completed' : ''}">
                        <div class="step-icon">📝</div>
                        <div class="step-info">
                            <span class="step-title">Enregistrée</span>
                            <span class="step-date">${date}</span>
                        </div>
                    </div>
                    
                    <div class="tracking-line ${statusInfo.step >= 2 ? 'completed' : ''}"></div>
                    
                    <div class="tracking-step ${statusInfo.step >= 2 ? 'completed' : ''}">
                        <div class="step-icon">🎨</div>
                        <div class="step-info">
                            <span class="step-title">Production</span>
                            <span class="step-date">${statusInfo.step >= 2 ? 'En cours' : '-'}</span>
                        </div>
                    </div>
                    
                    <div class="tracking-line ${statusInfo.step >= 3 ? 'completed' : ''}"></div>
                    
                    <div class="tracking-step ${statusInfo.step >= 3 ? 'completed' : ''}">
                        <div class="step-icon">🚚</div>
                        <div class="step-info">
                            <span class="step-title">Expédiée</span>
                            <span class="step-date">${order.date_expedition ? new Date(order.date_expedition).toLocaleDateString('fr-FR') : '-'}</span>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="order-card-footer">
                <button class="btn-order-details" onclick="toggleOrderDetails(${order.id})">
                    📋 Voir le détail
                </button>
            </div>
            
            <div class="order-details-container" id="order-details-${order.id}" style="display: none;">
                <div class="order-details-loading">
                    <div class="spinner"></div>
                </div>
            </div>
        </div>
    `;
}

// Toggle affichage des détails de commande
async function toggleOrderDetails(orderId) {
    const container = document.getElementById(`order-details-${orderId}`);
    const btn = document.querySelector(`#order-card-${orderId} .btn-order-details`);
    
    if (container.style.display === 'none') {
        container.style.display = 'block';
        btn.innerHTML = '📋 Masquer le détail';
        
        // Charger les détails si pas encore fait
        if (container.querySelector('.order-details-loading')) {
            await loadOrderDetails(orderId);
        }
    } else {
        container.style.display = 'none';
        btn.innerHTML = '📋 Voir le détail';
    }
}

// Charger les détails d'une commande
async function loadOrderDetails(orderId) {
    const container = document.getElementById(`order-details-${orderId}`);
    
    try {
        const response = await fetch(`${CONFIG.API_URL}?action=order_details&order_id=${orderId}`);
        const data = await response.json();
        
        if (!data.success) {
            throw new Error(data.error || 'Erreur');
        }
        
        if (!data.data || data.data.length === 0) {
            container.innerHTML = '<div class="no-products">Aucun produit trouvé</div>';
            return;
        }
        
        container.innerHTML = `
            <div class="order-products-list">
                ${data.data.map(product => renderProductItem(product)).join('')}
            </div>
        `;
        
    } catch (error) {
        console.error('Erreur chargement détails:', error);
        container.innerHTML = '<div class="order-details-error">Erreur de chargement</div>';
    }
}

// Rendu d'un produit de commande
function renderProductItem(product) {
    const price = parseFloat(product.price || 0).toFixed(2);
    
    return `
        <div class="order-product-item">
            <div class="product-image">
                ${product.image_url ? `<img src="${product.image_url}" alt="Produit">` : '<span class="no-image">🖼️</span>'}
            </div>
            <div class="product-info">
                <span class="product-style">${product.style || 'Style inconnu'}</span>
                <span class="product-format">${product.format || ''} ${product.delivery ? `• ${product.delivery}` : ''}</span>
            </div>
            <div class="product-qty">x${product.quantity || 1}</div>
            <div class="product-price">${price} €</div>
        </div>
    `;
}

// Déterminer le statut de la commande
function getOrderStatus(order) {
    const status = (order.statut_cmde || order.statut || 'cmde').toLowerCase();
    
    if (status === 'expedie' || status === 'expédié' || status === 'livre' || status === 'livré' || status === 'shipped') {
        return { step: 3, label: '🚚 Expédiée', class: 'status-shipped' };
    } else if (status === 'production' || status === 'en_production' || status === 'en cours' || status === 'processing') {
        return { step: 2, label: '🎨 En production', class: 'status-production' };
    } else {
        return { step: 1, label: '📝 Enregistrée', class: 'status-registered' };
    }
}

// Fermer manuellement le menu
function closeUserMenuManually() {
    const authButton = document.getElementById('authButton');
    if (authButton) {
        authButton.classList.remove('menu-open');
    }
}

// Gérer la déconnexion
function handleLogout() {
    closeUserMenuManually();
    if (confirm('Voulez-vous vraiment vous déconnecter ?')) {
        authManager.logout();
        
        // Vider les favoris
        if (typeof FavorisManager !== 'undefined') {
            FavorisManager.clear();
        }
        
        showToast('👋 Vous êtes déconnecté');
        updateAuthUI();
        
        // Retour à l'accueil
        if (typeof goHome === 'function') {
            goHome();
        }
    }
}

// Afficher le profil (redirection vers gestion du compte)
function showProfile() {
    showAccountSettings();
}

// Afficher la page de gestion du compte
function showAccountSettings() {
    closeUserMenuManually();
    
    // Supprimer une éventuelle modale existante
    const existingModal = document.getElementById('accountSettingsModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    const user = authManager.user;
    
    const modalHTML = `
        <div id="accountSettingsModal" class="auth-modal-overlay">
            <div class="auth-modal account-settings-modal">
                <button class="auth-modal-close" onclick="closeAccountSettings()">✕</button>
                
                <div class="auth-modal-header compact-header">
                    <h2>⚙️ Mon compte</h2>
                </div>
                
                <div id="accountFormContainer" class="account-form-container compact">
                    <form id="accountSettingsForm" onsubmit="handleUpdateAccount(event)">
                        
                        <div class="account-section">
                            <h3 class="account-section-title">👤 Informations</h3>
                            
                            <div class="auth-form-row">
                                <div class="auth-form-group compact">
                                    <label>Nom *</label>
                                    <input type="text" name="nom" required value="${user.nom || ''}" />
                                </div>
                                <div class="auth-form-group compact">
                                    <label>Prénom</label>
                                    <input type="text" name="prenom" value="${user.prenom || ''}" />
                                </div>
                            </div>
                            
                            <div class="auth-form-row">
                                <div class="auth-form-group compact">
                                    <label>Email *</label>
                                    <input type="email" name="email" required value="${user.email || ''}" />
                                </div>
                                <div class="auth-form-group compact">
                                    <label>Téléphone</label>
                                    <input type="tel" name="telephone" value="${user.telephone || ''}" placeholder="06 12 34 56 78" />
                                </div>
                            </div>
                        </div>
                        
                        <div class="account-section">
                            <h3 class="account-section-title">📍 Adresse</h3>
                            
                            <div class="auth-form-group compact">
                                <label>Adresse</label>
                                <input type="text" name="adresse" value="${user.adresse || ''}" placeholder="123 Rue de la Paix" />
                            </div>
                            
                            <div class="auth-form-row">
                                <div class="auth-form-group compact">
                                    <label>Ville</label>
                                    <input type="text" name="ville" value="${user.ville || ''}" placeholder="Paris" />
                                </div>
                                <div class="auth-form-group compact">
                                    <label>Code postal</label>
                                    <input type="text" name="code_postal" value="${user.code_postal || ''}" placeholder="75001" />
                                </div>
                            </div>
                        </div>
                        
                        <div id="accountError" class="auth-error" style="display: none;"></div>
                        <div id="accountSuccess" class="auth-success" style="display: none;"></div>
                        
                        <button type="submit" class="auth-btn-primary compact" id="saveAccountBtn">
                            💾 Enregistrer
                        </button>
                    </form>
                    
                    <div class="account-section account-password-section compact">
                        <button type="button" class="btn-change-password" onclick="togglePasswordForm()">
                            🔐 Modifier mon mot de passe
                        </button>
                        
                        <form id="passwordChangeForm" class="password-change-form" style="display: none;" onsubmit="handleChangePassword(event)">
                            <div class="auth-form-group compact">
                                <label>Mot de passe actuel</label>
                                <input type="password" name="current_password" required placeholder="••••••••" />
                            </div>
                            <div class="auth-form-group compact">
                                <label>Nouveau mot de passe</label>
                                <input type="password" name="new_password" required placeholder="••••••••" />
                            </div>
                            <div class="auth-form-group compact">
                                <label>Confirmer</label>
                                <input type="password" name="confirm_password" required placeholder="••••••••" />
                            </div>
                            <div id="passwordError" class="auth-error" style="display: none;"></div>
                            <button type="submit" class="auth-btn-secondary compact" id="changePasswordBtn">
                                Valider
                            </button>
                        </form>
                    </div>
                </div>
                
                <div class="modal-footer-actions">
                    <button type="button" class="btn-close-modal" onclick="closeAccountSettings()">
                        ✕ Fermer
                    </button>
                </div>
            </div>
        </div>
    `;
    
    document.body.insertAdjacentHTML('beforeend', modalHTML);
    
    // Gestionnaire de fermeture
    const overlay = document.getElementById('accountSettingsModal');
    const modal = overlay.querySelector('.auth-modal');
    
    let mouseDownTarget = null;
    
    overlay.addEventListener('mousedown', (e) => {
        mouseDownTarget = e.target;
    });
    
    overlay.addEventListener('click', (e) => {
        if (e.target === overlay && mouseDownTarget === overlay) {
            closeAccountSettings();
        }
    });
    
    modal.addEventListener('click', (e) => {
        e.stopPropagation();
    });
    
    // Touche Échap
    const handleEscape = (e) => {
        if (e.key === 'Escape') {
            closeAccountSettings();
            document.removeEventListener('keydown', handleEscape);
        }
    };
    document.addEventListener('keydown', handleEscape);
}

// Fermer la modale de gestion du compte
function closeAccountSettings() {
    const modal = document.getElementById('accountSettingsModal');
    if (modal) {
        modal.classList.add('closing');
        setTimeout(() => modal.remove(), 300);
    }
}

// Toggle du formulaire de changement de mot de passe
function togglePasswordForm() {
    const form = document.getElementById('passwordChangeForm');
    const btn = document.querySelector('.btn-change-password');
    
    if (form.style.display === 'none') {
        form.style.display = 'flex';
        btn.textContent = 'Annuler';
        btn.classList.add('cancel');
    } else {
        form.style.display = 'none';
        btn.textContent = 'Modifier mon mot de passe';
        btn.classList.remove('cancel');
        form.reset();
        document.getElementById('passwordError').style.display = 'none';
    }
}

// Gérer la mise à jour du compte
async function handleUpdateAccount(event) {
    event.preventDefault();
    
    const btn = document.getElementById('saveAccountBtn');
    const form = event.target;
    const formData = new FormData(form);
    
    const fields = {
        nom: formData.get('nom'),
        prenom: formData.get('prenom'),
        email: formData.get('email'),
        telephone: formData.get('telephone'),
        adresse: formData.get('adresse'),
        ville: formData.get('ville'),
        code_postal: formData.get('code_postal')
    };
    
    btn.disabled = true;
    btn.innerHTML = '<div class="spinner"></div> Enregistrement...';
    
    // Cacher les messages précédents
    document.getElementById('accountError').style.display = 'none';
    document.getElementById('accountSuccess').style.display = 'none';
    
    try {
        await authManager.updateProfile(fields);
        
        // Afficher le succès
        const successDiv = document.getElementById('accountSuccess');
        successDiv.textContent = '✅ Vos informations ont été mises à jour !';
        successDiv.style.display = 'block';
        
        // Mettre à jour l'UI du header
        updateAuthUI();
        
        showToast('✅ Profil mis à jour !');
        
    } catch (error) {
        const errorDiv = document.getElementById('accountError');
        errorDiv.textContent = error.message;
        errorDiv.style.display = 'block';
    } finally {
        btn.disabled = false;
        btn.innerHTML = '💾 Enregistrer les modifications';
    }
}

// Gérer le changement de mot de passe
async function handleChangePassword(event) {
    event.preventDefault();
    
    const btn = document.getElementById('changePasswordBtn');
    const form = event.target;
    const formData = new FormData(form);
    
    const currentPassword = formData.get('current_password');
    const newPassword = formData.get('new_password');
    const confirmPassword = formData.get('confirm_password');
    
    // Vérification
    if (newPassword !== confirmPassword) {
        const errorDiv = document.getElementById('passwordError');
        errorDiv.textContent = 'Les mots de passe ne correspondent pas';
        errorDiv.style.display = 'block';
        return;
    }
    
    btn.disabled = true;
    btn.innerHTML = '<div class="spinner"></div> Modification...';
    
    document.getElementById('passwordError').style.display = 'none';
    
    try {
        await authManager.changePassword(currentPassword, newPassword);
        
        showToast('✅ Mot de passe modifié !');
        
        // Fermer le formulaire
        togglePasswordForm();
        
    } catch (error) {
        const errorDiv = document.getElementById('passwordError');
        errorDiv.textContent = error.message;
        errorDiv.style.display = 'block';
    } finally {
        btn.disabled = false;
        btn.innerHTML = '🔐 Changer le mot de passe';
    }
}

// =====================================================
// INITIALISATION
// =====================================================

// Instance globale du gestionnaire d'authentification
let authManager;

// Initialiser au chargement de la page
document.addEventListener('DOMContentLoaded', async () => {
    authManager = new AuthManager();
    
    console.log('🔐 Auth Manager initialisé');
    console.log('Authentifié:', authManager.isAuthenticated);
    
    if (authManager.isAuthenticated) {
        console.log('👤 Utilisateur:', authManager.user.email);
        
        // Charger les favoris du client connecté
        if (typeof FavorisManager !== 'undefined') {
            await FavorisManager.init();
            console.log('❤️ Favoris chargés au démarrage');
        }
    }
    
    // Mettre à jour l'UI
    updateAuthUI();
});