/**
 * Application principale - Initialisation et démarrage
 */

const App = {
    /**
     * Initialise l'application
     */
    async init() {
        console.log('🚀 Initialisation de l\'application...');
        
        try {
            // 0. Initialiser la protection des images
            if (typeof ImageProtection !== 'undefined') {
                ImageProtection.init();
            }
            
            // 1. Initialiser l'état
            StateManager.init();
            
            // 2. Charger les tarifs depuis la BDD
            console.log('💰 Chargement des tarifs...');
            await this.loadTarifs();
            
            // 3. Charger les catégories
            console.log('📂 Chargement des catégories...');
            const categories = await ApiManager.loadCategories();
            StateManager.setState({ categories });
            
            // 4. Charger tous les styles
            console.log('🎨 Chargement des styles...');
            const allStyles = await ApiManager.loadAllStyles();
            StateManager.setState({ 
                allStyles,
                styles: allStyles  // Par défaut, afficher tous les styles
            });
            
            // 5. Mise à jour du compteur panier
            CartManager.updateCartCounter();
            
            // 6. Initialiser les favoris
            if (typeof FavorisManager !== 'undefined') {
                console.log('❤️ Chargement des favoris...');
                await FavorisManager.init();
            }
            
            // 7. Premier rendu
            console.log('🎨 Premier rendu...');
            RenderManager.render();
            
            // 8. Setup des écouteurs d'événements globaux
            this.setupGlobalListeners();
            
            console.log('✅ Application initialisée avec succès');
            
        } catch (error) {
            console.error('❌ Erreur lors de l\'initialisation:', error);
            Utils.showToast('❌ Erreur lors du chargement de l\'application');
            
            // Afficher une page d'erreur
            const app = document.getElementById('app');
            if (app) {
                app.innerHTML = `
                    <div class="error-page">
                        <div class="error-icon">⚠️</div>
                        <h1 class="error-title">Erreur de chargement</h1>
                        <p class="error-message">Impossible de charger l'application. Veuillez réessayer.</p>
                        <button class="btn btn-primary" onclick="location.reload()">
                            🔄 Recharger la page
                        </button>
                    </div>
                `;
            }
        }
    },
    
    /**
     * Configure les écouteurs d'événements globaux
     */
    setupGlobalListeners() {
        // Fermeture de la modale d'aperçu avec Escape
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape') {
                UIManager.closeImagePreview();
            }
        });
        
        // Prévenir la navigation accidentelle si génération en cours
        window.addEventListener('beforeunload', (e) => {
            const state = StateManager.getState();
            if (state.view === 'generating' || state.isSubmitting) {
                e.preventDefault();
                e.returnValue = '';
                return '';
            }
        });
        
        // Log des erreurs JavaScript
        window.addEventListener('error', (e) => {
            console.error('❌ Erreur globale:', e.error);
        });
        
        // Log des promesses rejetées
        window.addEventListener('unhandledrejection', (e) => {
            console.error('❌ Promise rejetée:', e.reason);
        });
    },
    
    /**
     * Charge les tarifs depuis la base de données
     */
    async loadTarifs() {
        try {
            const response = await fetch(`${CONFIG.API_URL}?action=tarifs`);
            const data = await response.json();
            
            if (data.success && data.data) {
                // Mettre à jour CONFIG.FORMATS avec les tarifs de la BDD
                CONFIG.FORMATS = data.data;
                console.log('💰 Tarifs chargés:', CONFIG.FORMATS);
            } else {
                console.warn('⚠️ Tarifs non chargés, utilisation des valeurs par défaut');
            }
        } catch (error) {
            console.error('❌ Erreur chargement tarifs:', error);
            // Garder les valeurs par défaut de CONFIG.FORMATS
        }
    },
    
    /**
     * Expose les fonctions globales pour les événements onclick
     */
    exposeGlobalFunctions() {
        // Navigation
        window.goHome = () => NavigationManager.goHome();
        window.filterByCategory = (slug) => NavigationManager.filterByCategory(slug);
        window.selectStyle = (style) => NavigationManager.selectStyle(style);
        window.backToCategories = () => NavigationManager.backToCategories();
        window.backToStyles = () => NavigationManager.backToStyles();
        
        // Cart
        window.viewCart = () => NavigationManager.viewCart();
        window.removeFromCart = (id) => {
            CartManager.removeFromCart(id);
            RenderManager.render();
        };
        window.updateCartQuantity = (id, qty) => {
            CartManager.updateQuantity(id, qty);
            RenderManager.render();
        };
        window.clearCart = () => CartManager.clearCart();
        
        // Order
        window.validateImage = () => NavigationManager.validateImage();
        window.validateAllImages = () => NavigationManager.validateAllImages();
        window.regenerate = () => NavigationManager.regenerate();
        window.selectFormat = (format) => {
            OrderManager.selectFormat(format);
            PricingManager.updateDisplay();
        };
        window.selectDelivery = (mode) => {
            OrderManager.selectDelivery(mode);
            PricingManager.updateDisplay();
        };
        window.selectQuantityFromGrid = (qty) => {
            OrderManager.selectQuantityFromGrid(qty);
            PricingManager.updateDisplay();
        };
        
        // Checkout
        window.proceedToCheckout = () => NavigationManager.proceedToCheckout();
        window.continueShopping = () => NavigationManager.continueShopping();
        
        // UI
        window.showImagePreview = (url, title) => UIManager.showImagePreview(url, title);
        window.closeImagePreview = (event) => UIManager.closeImagePreview(event);
        
        // Recherche
        window.performSearch = performSearch;
        window.handleSearchKeypress = handleSearchKeypress;
        
        console.log('✅ Fonctions globales exposées');
    }
};

// Fonction de recherche
async function performSearch() {
    const input = document.getElementById('searchInput');
    const query = input.value.trim();
    
    if (query.length < 2) {
        Utils.showToast('⚠️ Entrez au moins 2 caractères');
        return;
    }
    
    try {
        console.log('🔍 Recherche:', query);
        const results = await ApiManager.searchStyles(query);
        
        // Effacer le champ de recherche
        input.value = '';
        
        // Mettre à jour l'état avec les résultats
        StateManager.setState({
            view: 'homepage',
            styles: results,
            selectedCategory: null,
            searchQuery: query,
            showingFavoris: false  // Reset favoris
        });
        
        // Forcer le re-rendu
        RenderManager.render();
        
        if (results.length === 0) {
            Utils.showToast('🔍 Aucun résultat pour "' + query + '"');
        } else {
            Utils.showToast('🔍 ' + results.length + ' résultat(s) trouvé(s)');
        }
        
    } catch (error) {
        console.error('❌ Erreur recherche:', error);
        Utils.showToast('❌ Erreur lors de la recherche');
    }
}

// Gestion de la touche Entrée
function handleSearchKeypress(event) {
    if (event.key === 'Enter') {
        performSearch();
    }
}

// Inscription à la newsletter
async function subscribeNewsletter(event) {
    event.preventDefault();
    
    const emailInput = document.getElementById('newsletterEmail');
    const btn = document.getElementById('newsletterBtn');
    const messageDiv = document.getElementById('newsletterMessage');
    const form = document.getElementById('newsletterForm');
    
    const email = emailInput.value.trim();
    
    if (!email) {
        Utils.showToast('⚠️ Veuillez entrer votre email');
        return false;
    }
    
    // Désactiver le bouton pendant l'envoi
    btn.disabled = true;
    btn.textContent = '⏳ Inscription...';
    
    try {
        // Récupérer le client_id si connecté
        let clientId = null;
        if (typeof authManager !== 'undefined' && authManager.isAuthenticated && authManager.user) {
            clientId = authManager.user.client_id;
        }
        
        const response = await fetch(`${CONFIG.API_URL}?action=newsletter_subscribe`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                email: email,
                client_id: clientId
            })
        });
        
        const data = await response.json();
        
        if (data.success) {
            // Afficher le message de succès
            form.style.display = 'none';
            messageDiv.style.display = 'block';
            messageDiv.innerHTML = `<span class="newsletter-success">✅ ${data.message}</span>`;
            
            Utils.showToast('✅ ' + data.message);
        } else {
            throw new Error(data.error || 'Erreur lors de l\'inscription');
        }
        
    } catch (error) {
        console.error('❌ Erreur newsletter:', error);
        Utils.showToast('❌ ' + error.message);
        
        // Réactiver le bouton
        btn.disabled = false;
        btn.textContent = 'S\'inscrire';
    }
    
    return false;
}

// Démarrage de l'application au chargement du DOM
document.addEventListener('DOMContentLoaded', () => {
    App.exposeGlobalFunctions();
    App.init();
});