<?php
/**
 * API AUTHENTIFICATION - GESTION DES COMPTES CLIENTS
 * Inscription, connexion, profil, réinitialisation
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Gestion des requêtes OPTIONS (CORS)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Configuration BDD (identique à api.php)
define('DB_HOST', 'oz9643-001.eu.clouddb.ovh.net');
define('DB_PORT', '35297');
define('DB_USER', 'user_recreatoon');
define('DB_PASS', 'MathelodIMG59');  // <-- REMPLACEZ PAR VOTRE MOT DE PASSE
define('DB_NAME', 'recreatoon_DDB');

// Configuration JWT (à personnaliser)
define('JWT_SECRET', 'VotreCleSuperSecreteAChanger2024!CaCartoon');
define('JWT_EXPIRATION', 7 * 24 * 3600); // 7 jours

// Connexion PDO
function getDB() {
    static $pdo = null;
    if ($pdo === null) {
        try {
            $dsn = sprintf("pgsql:host=%s;port=%s;dbname=%s", DB_HOST, DB_PORT, DB_NAME);
            $pdo = new PDO($dsn, DB_USER, DB_PASS, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false
            ]);
        } catch(PDOException $e) {
            http_response_code(500);
            die(json_encode(['success' => false, 'error' => 'Connexion base de données échouée']));
        }
    }
    return $pdo;
}

// Fonction pour générer un JWT simple
function generateJWT($userId, $email) {
    $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
    $payload = json_encode([
        'user_id' => $userId,
        'email' => $email,
        'iat' => time(),
        'exp' => time() + JWT_EXPIRATION
    ]);
    
    $base64UrlHeader = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
    $base64UrlPayload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));
    
    $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, JWT_SECRET, true);
    $base64UrlSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
    
    return $base64UrlHeader . "." . $base64UrlPayload . "." . $base64UrlSignature;
}

// Fonction pour vérifier un JWT
function verifyJWT($token) {
    $parts = explode('.', $token);
    if (count($parts) !== 3) {
        return false;
    }
    
    list($header, $payload, $signature) = $parts;
    
    $validSignature = hash_hmac('sha256', $header . "." . $payload, JWT_SECRET, true);
    $validSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($validSignature));
    
    if ($signature !== $validSignature) {
        return false;
    }
    
    $payload = json_decode(base64_decode(str_replace(['-', '_'], ['+', '/'], $payload)), true);
    
    if (!$payload || $payload['exp'] < time()) {
        return false;
    }
    
    return $payload;
}

// Récupérer le token depuis les headers
function getBearerToken() {
    // Essayer depuis les headers
    $headers = getallheaders();
    
    // getallheaders peut être sensible à la casse selon les serveurs
    foreach ($headers as $key => $value) {
        if (strtolower($key) === 'authorization') {
            if (preg_match('/Bearer\s+(.*)$/i', $value, $matches)) {
                return $matches[1];
            }
        }
    }
    
    // Fallback : lire depuis $_SERVER
    if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
        if (preg_match('/Bearer\s+(.*)$/i', $_SERVER['HTTP_AUTHORIZATION'], $matches)) {
            return $matches[1];
        }
    }
    
    // Autre fallback pour Apache
    if (isset($_SERVER['REDIRECT_HTTP_AUTHORIZATION'])) {
        if (preg_match('/Bearer\s+(.*)$/i', $_SERVER['REDIRECT_HTTP_AUTHORIZATION'], $matches)) {
            return $matches[1];
        }
    }
    
    return null;
}

// Obtenir l'IP du client
function getClientIP() {
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
        return $_SERVER['HTTP_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        return $_SERVER['HTTP_X_FORWARDED_FOR'];
    } else {
        return $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    }
}

// Validation email
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

// Validation mot de passe (min 8 caractères, 1 majuscule, 1 minuscule, 1 chiffre)
function validatePassword($password) {
    return strlen($password) >= 8 
        && preg_match('/[A-Z]/', $password) 
        && preg_match('/[a-z]/', $password) 
        && preg_match('/[0-9]/', $password);
}

// Router
$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

try {
    $pdo = getDB();
    
    // ========================================
    // POST /auth.php?action=register
    // INSCRIPTION D'UN NOUVEAU CLIENT
    // ========================================
    if ($method === 'POST' && $action === 'register') {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $email = trim($data['email'] ?? '');
        $password = $data['password'] ?? '';
        $nom = trim($data['nom'] ?? '');
        $prenom = trim($data['prenom'] ?? '');
        $telephone = trim($data['telephone'] ?? '');
        $adresse = trim($data['adresse'] ?? '');
        $ville = trim($data['ville'] ?? '');
        $code_postal = trim($data['code_postal'] ?? '');
        
        // Validations
        if (empty($email) || !validateEmail($email)) {
            throw new Exception('Email invalide');
        }
        
        if (empty($password) || !validatePassword($password)) {
            throw new Exception('Le mot de passe doit contenir au moins 8 caractères, une majuscule, une minuscule et un chiffre');
        }
        
        if (empty($nom)) {
            throw new Exception('Le nom est requis');
        }
        
        // Vérifier si l'email existe déjà
        $stmt = $pdo->prepare("SELECT id FROM client_affiche WHERE email = ?");
        $stmt->execute([$email]);
        if ($stmt->fetch()) {
            throw new Exception('Cet email est déjà utilisé');
        }
        
        // Hasher le mot de passe
        $passwordHash = password_hash($password, PASSWORD_BCRYPT);
        
        // Insérer le client
        $stmt = $pdo->prepare("
            INSERT INTO client_affiche (
                email, password_hash, nom, prenom, telephone,
                adresse, ville, code_postal,
                ip_creation, date_derniere_connexion
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP)
            RETURNING id, email, nom, prenom, telephone, adresse, ville, code_postal
        ");
        
        $stmt->execute([
            $email,
            $passwordHash,
            $nom,
            $prenom,
            $telephone,
            $adresse,
            $ville,
            $code_postal,
            getClientIP()
        ]);
        
        $user = $stmt->fetch();
        
        // Générer le token JWT
        $token = generateJWT($user['id'], $user['email']);
        
        echo json_encode([
            'success' => true,
            'message' => 'Inscription réussie',
            'data' => [
                'user' => $user,
                'token' => $token
            ]
        ]);
    }
    
    // ========================================
    // POST /auth.php?action=login
    // CONNEXION D'UN CLIENT
    // ========================================
    elseif ($method === 'POST' && $action === 'login') {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $email = trim($data['email'] ?? '');
        $password = $data['password'] ?? '';
        
        if (empty($email) || empty($password)) {
            throw new Exception('Email et mot de passe requis');
        }
        
        // Récupérer le client
        $stmt = $pdo->prepare("
            SELECT id, email, password_hash, nom, prenom, telephone, 
                   adresse, ville, code_postal, actif
            FROM client_affiche 
            WHERE email = ?
        ");
        $stmt->execute([$email]);
        $user = $stmt->fetch();
        
        if (!$user || !password_verify($password, $user['password_hash'])) {
            throw new Exception('Email ou mot de passe incorrect');
        }
        
        if (!$user['actif']) {
            throw new Exception('Votre compte a été désactivé');
        }
        
        // Mettre à jour la dernière connexion
        $stmt = $pdo->prepare("
            UPDATE client_affiche 
            SET date_derniere_connexion = CURRENT_TIMESTAMP,
                ip_derniere_connexion = ?
            WHERE id = ?
        ");
        $stmt->execute([getClientIP(), $user['id']]);
        
        // Retirer le hash du password
        unset($user['password_hash']);
        unset($user['actif']);
        
        // Générer le token JWT
        $token = generateJWT($user['id'], $user['email']);
        
        echo json_encode([
            'success' => true,
            'message' => 'Connexion réussie',
            'data' => [
                'user' => $user,
                'token' => $token
            ]
        ]);
    }
    
    // ========================================
    // GET /auth.php?action=profile
    // RÉCUPÉRER LE PROFIL DU CLIENT CONNECTÉ
    // ========================================
    elseif ($method === 'GET' && $action === 'profile') {
        $token = getBearerToken();
        if (!$token) {
            throw new Exception('Token manquant');
        }
        
        $payload = verifyJWT($token);
        if (!$payload) {
            throw new Exception('Token invalide ou expiré');
        }
        
        $stmt = $pdo->prepare("
            SELECT id, email, nom, prenom, telephone, 
                   adresse, ville, code_postal, pays,
                   newsletter, nb_commandes, montant_total_achats,
                   date_creation, date_derniere_connexion
            FROM client_affiche 
            WHERE id = ? AND actif = TRUE
        ");
        $stmt->execute([$payload['user_id']]);
        $user = $stmt->fetch();
        
        if (!$user) {
            throw new Exception('Utilisateur non trouvé');
        }
        
        echo json_encode([
            'success' => true,
            'data' => $user
        ]);
    }
    
    // ========================================
    // PUT /auth.php?action=update
    // METTRE À JOUR LE PROFIL
    // ========================================
    elseif ($method === 'PUT' && $action === 'update') {
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Récupérer le token depuis header OU body
        $token = getBearerToken();
        if (!$token && isset($data['token'])) {
            $token = $data['token'];
        }
        
        if (!$token) {
            throw new Exception('Token manquant');
        }
        
        $payload = verifyJWT($token);
        if (!$payload) {
            throw new Exception('Token invalide ou expiré');
        }
        
        $fields = [];
        $values = [];
        
        if (isset($data['nom'])) {
            $fields[] = "nom = ?";
            $values[] = trim($data['nom']);
        }
        if (isset($data['prenom'])) {
            $fields[] = "prenom = ?";
            $values[] = trim($data['prenom']);
        }
        if (isset($data['telephone'])) {
            $fields[] = "telephone = ?";
            $values[] = trim($data['telephone']);
        }
        if (isset($data['adresse'])) {
            $fields[] = "adresse = ?";
            $values[] = trim($data['adresse']);
        }
        if (isset($data['ville'])) {
            $fields[] = "ville = ?";
            $values[] = trim($data['ville']);
        }
        if (isset($data['code_postal'])) {
            $fields[] = "code_postal = ?";
            $values[] = trim($data['code_postal']);
        }
        if (isset($data['newsletter'])) {
            $fields[] = "newsletter = ?";
            $values[] = (bool)$data['newsletter'];
        }
        
        if (empty($fields)) {
            throw new Exception('Aucune donnée à mettre à jour');
        }
        
        $values[] = $payload['user_id'];
        
        $sql = "UPDATE client_affiche SET " . implode(", ", $fields) . " WHERE id = ? RETURNING *";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($values);
        
        $user = $stmt->fetch();
        unset($user['password_hash']);
        
        echo json_encode([
            'success' => true,
            'message' => 'Profil mis à jour',
            'data' => $user
        ]);
    }
    
    // ========================================
    // POST /auth.php?action=verify-token
    // VÉRIFIER LA VALIDITÉ D'UN TOKEN
    // ========================================
    elseif ($method === 'POST' && $action === 'verify-token') {
        $data = json_decode(file_get_contents('php://input'), true);
        $token = $data['token'] ?? '';
        
        $payload = verifyJWT($token);
        
        if (!$payload) {
            throw new Exception('Token invalide');
        }
        
        echo json_encode([
            'success' => true,
            'data' => [
                'valid' => true,
                'user_id' => $payload['user_id'],
                'email' => $payload['email']
            ]
        ]);
    }
    
    // ========================================
    // POST /auth.php?action=logout
    // DÉCONNEXION (côté client surtout)
    // ========================================
    elseif ($method === 'POST' && $action === 'logout') {
        echo json_encode([
            'success' => true,
            'message' => 'Déconnexion réussie'
        ]);
    }
    
    // ========================================
    // POST /auth.php?action=change-password
    // CHANGER LE MOT DE PASSE
    // ========================================
    elseif ($method === 'POST' && $action === 'change-password') {
        $token = getBearerToken();
        if (!$token) {
            throw new Exception('Token manquant');
        }
        
        $payload = verifyJWT($token);
        if (!$payload) {
            throw new Exception('Token invalide ou expiré');
        }
        
        $data = json_decode(file_get_contents('php://input'), true);
        $currentPassword = $data['current_password'] ?? '';
        $newPassword = $data['new_password'] ?? '';
        
        if (empty($currentPassword) || empty($newPassword)) {
            throw new Exception('Mots de passe requis');
        }
        
        if (!validatePassword($newPassword)) {
            throw new Exception('Le nouveau mot de passe doit contenir au moins 8 caractères, une majuscule, une minuscule et un chiffre');
        }
        
        // Vérifier l'ancien mot de passe
        $stmt = $pdo->prepare("SELECT password_hash FROM client_affiche WHERE id = ?");
        $stmt->execute([$payload['user_id']]);
        $user = $stmt->fetch();
        
        if (!$user || !password_verify($currentPassword, $user['password_hash'])) {
            throw new Exception('Mot de passe actuel incorrect');
        }
        
        // Mettre à jour le mot de passe
        $newHash = password_hash($newPassword, PASSWORD_BCRYPT);
        $stmt = $pdo->prepare("UPDATE client_affiche SET password_hash = ? WHERE id = ?");
        $stmt->execute([$newHash, $payload['user_id']]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Mot de passe modifié avec succès'
        ]);
    }
    
    else {
        throw new Exception('Action invalide ou méthode non autorisée');
    }
    
} catch(Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}